/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.esci.adapter.map.order;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.math.Currencies;
import org.openvpms.archetype.rules.math.Currency;
import org.openvpms.archetype.rules.party.ContactArchetypes;
import org.openvpms.archetype.rules.party.Contacts;
import org.openvpms.archetype.rules.party.PartyRules;
import org.openvpms.archetype.rules.practice.LocationRules;
import org.openvpms.archetype.rules.practice.PracticeRules;
import org.openvpms.archetype.rules.supplier.SupplierRules;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityRelationship;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.esci.adapter.i18n.ESCIAdapterMessages;
import org.openvpms.esci.adapter.map.UBLHelper;
import org.openvpms.esci.adapter.util.ESCIAdapterException;
import org.openvpms.esci.ubl.common.aggregate.AddressLineType;
import org.openvpms.esci.ubl.common.aggregate.AddressType;
import org.openvpms.esci.ubl.common.aggregate.ContactType;
import org.openvpms.esci.ubl.common.aggregate.CustomerPartyType;
import org.openvpms.esci.ubl.common.aggregate.ItemIdentificationType;
import org.openvpms.esci.ubl.common.aggregate.ItemType;
import org.openvpms.esci.ubl.common.aggregate.LineItemType;
import org.openvpms.esci.ubl.common.aggregate.MonetaryTotalType;
import org.openvpms.esci.ubl.common.aggregate.OrderLineType;
import org.openvpms.esci.ubl.common.aggregate.PartyNameType;
import org.openvpms.esci.ubl.common.aggregate.PartyType;
import org.openvpms.esci.ubl.common.aggregate.PriceType;
import org.openvpms.esci.ubl.common.aggregate.SupplierPartyType;
import org.openvpms.esci.ubl.common.aggregate.TaxTotalType;
import org.openvpms.esci.ubl.common.basic.BaseQuantityType;
import org.openvpms.esci.ubl.common.basic.CityNameType;
import org.openvpms.esci.ubl.common.basic.CopyIndicatorType;
import org.openvpms.esci.ubl.common.basic.CountrySubentityType;
import org.openvpms.esci.ubl.common.basic.CustomerAssignedAccountIDType;
import org.openvpms.esci.ubl.common.basic.DescriptionType;
import org.openvpms.esci.ubl.common.basic.ElectronicMailType;
import org.openvpms.esci.ubl.common.basic.IDType;
import org.openvpms.esci.ubl.common.basic.IssueDateType;
import org.openvpms.esci.ubl.common.basic.IssueTimeType;
import org.openvpms.esci.ubl.common.basic.LineExtensionAmountType;
import org.openvpms.esci.ubl.common.basic.LineType;
import org.openvpms.esci.ubl.common.basic.NameType;
import org.openvpms.esci.ubl.common.basic.PackQuantityType;
import org.openvpms.esci.ubl.common.basic.PackSizeNumericType;
import org.openvpms.esci.ubl.common.basic.PayableAmountType;
import org.openvpms.esci.ubl.common.basic.PostalZoneType;
import org.openvpms.esci.ubl.common.basic.PriceAmountType;
import org.openvpms.esci.ubl.common.basic.QuantityType;
import org.openvpms.esci.ubl.common.basic.SupplierAssignedAccountIDType;
import org.openvpms.esci.ubl.common.basic.TaxAmountType;
import org.openvpms.esci.ubl.common.basic.TelefaxType;
import org.openvpms.esci.ubl.common.basic.TelephoneType;
import org.openvpms.esci.ubl.common.basic.TotalTaxAmountType;
import org.openvpms.esci.ubl.common.basic.UBLVersionIDType;
import org.openvpms.esci.ubl.order.Order;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import java.math.BigDecimal;
import java.util.GregorianCalendar;

import static java.math.BigDecimal.ONE;
import static java.math.BigDecimal.ZERO;
import static org.openvpms.archetype.rules.party.ContactArchetypes.BILLING_PURPOSE;
import static org.openvpms.archetype.rules.party.ContactArchetypes.FAX_PURPOSE;


/**
 * Maps <em>act.supplierOrder</em> acts to UBL Orders.
 *
 * @author Tim Anderson
 * @author bcharlton(benjicharlton @ gmail.com)
 */
public class OrderMapperImpl implements OrderMapper {

    /**
     * XML data type factory.
     */
    private final DatatypeFactory datatypeFactory;

    /**
     * The lookup service.
     */
    private final LookupService lookupService;

    /**
     * The practice rules.
     */
    private final PracticeRules practiceRules;

    /**
     * Location rules.
     */
    private final LocationRules locationRules;

    /**
     * Party rules.
     */
    private final PartyRules partyRules;

    /**
     * The supplier rules.
     */
    private final SupplierRules supplierRules;

    /**
     * The currencies.
     */
    private final Currencies currencies;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * Contact helper.
     */
    private final Contacts contacts;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(OrderMapperImpl.class);

    /**
     * Default package unit code, if none is specified. This corresponds to "each" in UNE/ECE rec 20
     * (http://docs.oasis-open.org/ubl/cs-UBL-2.0/cl/gc/cefact/UnitOfMeasureCode-2.0.gc)
     */
    private static final String DEFAULT_PACKAGE_UNITS = "EA";


    /**
     * Constructs an {@link OrderMapperImpl}.
     *
     * @param practiceRules the practice rules
     * @param locationRules the location rules
     * @param partyRules    the party rules
     * @param supplierRules the supplier rules
     * @param lookupService the lookup service
     * @param currencies    the currencies
     * @param service       the archetype service
     */
    public OrderMapperImpl(PracticeRules practiceRules, LocationRules locationRules, PartyRules partyRules,
                           SupplierRules supplierRules, LookupService lookupService, Currencies currencies,
                           ArchetypeService service) {
        try {
            datatypeFactory = DatatypeFactory.newInstance();
        } catch (DatatypeConfigurationException exception) {
            throw new IllegalStateException(exception);
        }
        this.practiceRules = practiceRules;
        this.locationRules = locationRules;
        this.partyRules = partyRules;
        this.supplierRules = supplierRules;
        this.lookupService = lookupService;
        this.currencies = currencies;
        this.service = service;
        contacts = new Contacts(service);
    }

    /**
     * Maps an <em>act.supplierOrder</em> to an UBL order.
     *
     * @param order the <em>act.supplierOrder</em> to map
     * @return the corresponding UBL order
     * @throws ESCIAdapterException      for mapping errors
     * @throws ArchetypeServiceException for any archetype service error
     */
    public Order map(FinancialAct order) {
        Order result = new Order();
        Currency currency = getCurrency();

        UBLVersionIDType version = UBLHelper.initID(new UBLVersionIDType(), "2.0");
        IDType id = UBLHelper.createID(order.getId());
        CopyIndicatorType copyIndicator = getCopyIndicatorType(false);

        GregorianCalendar startTime = new GregorianCalendar();
        startTime.setTime(order.getActivityStartTime());
        IssueDateType issueDate = UBLHelper.createIssueDate(startTime, datatypeFactory);
        IssueTimeType issueTime = UBLHelper.createIssueTime(startTime, datatypeFactory);

        IMObjectBean bean = service.getBean(order);
        Entity author = bean.getObject("createdBy", Entity.class);
        Party stockLocation = bean.getTarget("stockLocation", Party.class);
        Party location = getLocation(stockLocation);

        Party supplier = bean.getTarget("supplier", Party.class);
        EntityRelationship supplierStockLocation = supplierRules.getSupplierStockLocation(supplier, stockLocation);
        if (supplierStockLocation == null) {
            throw new ESCIAdapterException(ESCIAdapterMessages.ESCINotConfigured(supplier, stockLocation));
        }
        String contactName = (author != null) ? author.getName() : null;
        CustomerPartyType customerParty = getCustomer(contactName, location, stockLocation, supplierStockLocation);
        SupplierPartyType supplierParty = getSupplier(supplier);

        TaxTotalType taxTotal = getTaxTotal(order, currency);
        MonetaryTotalType total = getMonetaryTotal(order, currency);

        result.setUBLVersionID(version);
        result.setID(id);
        result.setCopyIndicator(copyIndicator);
        result.setIssueDate(issueDate);
        result.setIssueTime(issueTime);
        result.setBuyerCustomerParty(customerParty);
        result.setSellerSupplierParty(supplierParty);
        result.getTaxTotal().add(taxTotal);
        result.setAnticipatedMonetaryTotal(total);

        for (Act item : bean.getTargets("items", Act.class)) {
            OrderLineType line = getOrderLine(item, supplier, currency);
            result.getOrderLine().add(line);
        }
        return result;
    }

    /**
     * Returns an {@code OrderLineType} for an <em>act.supplierOrderItem</em>.
     *
     * @param act      the order item to map
     * @param supplier the supplier
     * @param currency the currency that amounts are expressed in
     * @return a new {@code OrderLineType} corresponding to the act
     */
    private OrderLineType getOrderLine(Act act, Party supplier, Currency currency) {
        IMObjectBean bean = service.getBean(act);
        Product product = bean.getTarget("product", Product.class);

        OrderLineType orderLine = new OrderLineType();
        LineItemType lineItem = new LineItemType();

        String packageUnits = bean.getString("packageUnits");
        String unitCode = getUnitCode(packageUnits);

        ItemType item = getItem(bean, supplier, product, unitCode);
        lineItem.setItem(item);
        orderLine.setLineItem(lineItem);

        IDType id = UBLHelper.createID(act.getId());
        QuantityType quantity = UBLHelper.initQuantity(new QuantityType(), bean.getBigDecimal("quantity"), unitCode);
        LineExtensionAmountType lineAmount
                = UBLHelper.initAmount(new LineExtensionAmountType(), bean.getBigDecimal("total"), currency);
        TotalTaxAmountType taxAmount
                = UBLHelper.initAmount(new TotalTaxAmountType(), bean.getBigDecimal("tax"), currency);
        PriceType price = getPrice(bean.getBigDecimal("unitPrice", ZERO), unitCode, currency);

        lineItem.setID(id);
        lineItem.setQuantity(quantity);
        lineItem.setLineExtensionAmount(lineAmount);
        lineItem.setTotalTaxAmount(taxAmount);
        lineItem.setPrice(price);

        return orderLine;
    }

    /**
     * Returns a {@code ItemType} for a supplier, order item, and product.
     *
     * @param bean         the order item
     * @param supplier     the supplier
     * @param product      the product
     * @param packageUnits the package size unit code. May be {@code null}
     * @return an {@code ItemType} corresponding to the supplier and product
     */
    private ItemType getItem(IMObjectBean bean, Party supplier, Product product, String packageUnits) {
        ItemType result = new ItemType();
        ItemIdentificationType buyersId = getItemIdentification(product.getId());
        String reorderCode = bean.getString("reorderCode");
        String reorderDescription = bean.getString("reorderDescription");
        if (!StringUtils.isEmpty(reorderCode)) {
            ItemIdentificationType sellersId = getItemIdentification(reorderCode);
            result.setSellersItemIdentification(sellersId);
        } else {
            throw new ESCIAdapterException(ESCIAdapterMessages.noSupplierOrderCode(supplier, product));
        }
        if (!StringUtils.isEmpty(reorderDescription)) {
            DescriptionType description = UBLHelper.initText(new DescriptionType(), reorderDescription);
            result.getDescription().add(description);
        }
        NameType name = UBLHelper.initName(new NameType(), product.getName());

        result.setBuyersItemIdentification(buyersId);
        result.setName(name);

        BigDecimal packageSize = bean.getBigDecimal("packageSize");
        if (packageSize != null && packageUnits != null) {
            PackQuantityType quantity = UBLHelper.initQuantity(new PackQuantityType(), ONE, packageUnits);
            PackSizeNumericType size = UBLHelper.createPackSizeNumeric(packageSize);
            result.setPackQuantity(quantity);
            result.setPackSizeNumeric(size);
        }

        return result;
    }

    /**
     * Returns a {@code PriceType} for the specified price and unit code.
     *
     * @param price    the price
     * @param unitCode the quantity unit code (UN/CEFACT). May be {@code null}
     * @param currency the currency
     * @return the corresponding {@code PriceType} for price and unitCode
     */
    private PriceType getPrice(BigDecimal price, String unitCode, Currency currency) {
        PriceType result = new PriceType();
        PriceAmountType priceAmount = UBLHelper.initAmount(new PriceAmountType(), price, currency);
        result.setPriceAmount(priceAmount);
        result.setBaseQuantity(UBLHelper.initQuantity(new BaseQuantityType(), ONE, unitCode));
        return result;
    }

    /**
     * Returns a {@code TaxTotalType} for an order.
     *
     * @param order    the order
     * @param currency the currency
     * @return the corresponding {@code TaxTotalType}
     */
    private TaxTotalType getTaxTotal(FinancialAct order, Currency currency) {
        TaxTotalType result = new TaxTotalType();
        result.setTaxAmount(UBLHelper.initAmount(new TaxAmountType(), order.getTaxAmount(), currency));
        return result;
    }

    /**
     * Returns a {@code MonetaryTotalType} for an order.
     *
     * @param order    the order
     * @param currency the currency
     * @return the corresponding {@code MonetaryTotalType}
     */
    private MonetaryTotalType getMonetaryTotal(FinancialAct order, Currency currency) {
        BigDecimal payableAmount = order.getTotal();
        BigDecimal lineExtensionAmount = payableAmount.subtract(order.getTaxAmount());

        MonetaryTotalType result = new MonetaryTotalType();
        result.setLineExtensionAmount(UBLHelper.initAmount(new LineExtensionAmountType(), lineExtensionAmount,
                                                           currency));
        result.setPayableAmount(UBLHelper.initAmount(new PayableAmountType(), payableAmount, currency));
        return result;
    }

    /**
     * Returns the UN/CEFACT unit code for the given package units code from an <em>lookup.uom</em>.
     * <p/>
     * If no package is specified, defaults to {@link #DEFAULT_PACKAGE_UNITS}.
     *
     * @param packageUnits the package units code
     * @return the corresponding unit code
     */
    private String getUnitCode(String packageUnits) {
        String result = null;
        if (!StringUtils.isEmpty("packageUnits")) {
            Lookup lookup = lookupService.getLookup("lookup.uom", packageUnits);
            if (lookup != null) {
                IMObjectBean lookupBean = service.getBean(lookup);
                String unitCode = lookupBean.getString("unitCode");
                if (!StringUtils.isEmpty(unitCode)) {
                    result = unitCode;
                }
            }
            if (result == null) {
                log.warn("No unit code for package units={}. Defaulting to {}", packageUnits, DEFAULT_PACKAGE_UNITS);
            }
        }
        if (result == null) {
            result = DEFAULT_PACKAGE_UNITS;
        }
        return result;
    }

    /**
     * Helper to return the location associated with a stock location.
     *
     * @param stockLocation the stock location
     * @return the corresponding location
     * @throws ESCIAdapterException if the stock location isn't associated with a practice location
     */
    private Party getLocation(Party stockLocation) {
        IMObjectBean bean = service.getBean(stockLocation);
        // TODO - there could be more than one location which refers to different party.organisationLocation 
        Party result = bean.getSource("locations", Party.class);
        if (result == null) {
            throw new ESCIAdapterException(ESCIAdapterMessages.noPracticeLocationForStockLocation(stockLocation));
        }
        return result;
    }

    /**
     * Returns a {@code CustomerPartyType} corresponding to the passed <em>party.organisationStockLocation</em>.
     * <p/>
     * The contact details will be either those of the <em>party.organisationLocation</em> or the parent
     * </em>party.organisationPractice</em>. If the location has a <em>contact.location</em>, then the location's
     * details will be used, otherwise the practice's details will be used.
     * <p/>
     * The customer identifier will be that of the stock location.
     * <p/>
     * NOTE: the supplied <em>entityRelationship.supplierStockLocation*</em> relationship may have an optional
     * <em>accountId</em> node, used to populate the {@code SupplierAssignedAccountIDType}
     *
     * @param contactName           a contact name to supply with telephone, email and fax details, May be {@code null}
     * @param location              the practice location
     * @param stockLocation         the stock location
     * @param supplierStockLocation an <em>entityRelationship.supplierStockLocation*</em> relationship
     * @return the corresponding {@code CustomerPartyType}
     */
    private CustomerPartyType getCustomer(String contactName, Party location, Party stockLocation,
                                          EntityRelationship supplierStockLocation) {
        CustomerPartyType result = new CustomerPartyType();
        Party customer;

        Party practice = locationRules.getPractice(location);
        if (practice == null) {
            throw new IllegalStateException("No practice for location: " + location.getId());
        }

        Contact locationContact = partyRules.getContact(location, ContactArchetypes.LOCATION, BILLING_PURPOSE);
        if (locationContact == null) {
            locationContact = partyRules.getContact(practice, ContactArchetypes.LOCATION, BILLING_PURPOSE);
            if (locationContact == null) {
                throw new IllegalStateException("No contact.location for location: " + location.getId());
            }
            customer = practice;
        } else {
            customer = location;
        }
        Contact phoneContact = partyRules.getContact(customer, ContactArchetypes.PHONE, false, FAX_PURPOSE,
                                                     BILLING_PURPOSE);
        Contact faxContact = partyRules.getContact(customer, ContactArchetypes.PHONE, true, null, FAX_PURPOSE,
                                                   BILLING_PURPOSE);
        if (faxContact == null) {
            faxContact = partyRules.getContact(customer, ContactArchetypes.PHONE, true, null, FAX_PURPOSE);
        }
        Contact emailContact = partyRules.getContact(customer, ContactArchetypes.EMAIL, BILLING_PURPOSE);

        CustomerAssignedAccountIDType customerId
                = UBLHelper.initID(new CustomerAssignedAccountIDType(), stockLocation.getId());

        PartyType party = getParty(customer, locationContact);
        party.setContact(getContact(contactName, phoneContact, faxContact, emailContact));

        result.setCustomerAssignedAccountID(customerId);

        IMObjectBean bean = service.getBean(supplierStockLocation);
        String accountId = bean.getString("accountId");
        if (!StringUtils.isEmpty(accountId)) {
            SupplierAssignedAccountIDType supplierId = UBLHelper.initID(new SupplierAssignedAccountIDType(), accountId);
            result.setSupplierAssignedAccountID(supplierId);
        }

        result.setParty(party);
        return result;
    }

    /**
     * Returns a {@code SupplierPartyType} corresponding to the passed supplier.
     *
     * @param supplier the supplier
     * @return the corresponding {@code SupplierPartyType}
     */
    private SupplierPartyType getSupplier(Party supplier) {
        SupplierPartyType result = new SupplierPartyType();

        CustomerAssignedAccountIDType accountId
                = UBLHelper.initID(new CustomerAssignedAccountIDType(), supplier.getId());
        Contact contact = partyRules.getContact(supplier, ContactArchetypes.LOCATION, null);

        result.setCustomerAssignedAccountID(accountId);
        result.setParty(getParty(supplier, contact));
        return result;
    }

    /**
     * Returns a {@code PartyType} for the supplied party and contact.
     *
     * @param party    the party
     * @param location the location contact. May be {@code null}
     * @return the corresponding {@code PartyType}
     */
    private PartyType getParty(Party party, Contact location) {
        PartyType result = new PartyType();

        PartyNameType partyName = new PartyNameType();
        partyName.setName(UBLHelper.createName(party.getName()));
        result.getPartyName().add(partyName);
        if (location != null) {
            result.setPostalAddress(getAddress(location));
        }
        return result;
    }

    /**
     * Returns a {@code ContactType} for the supplied contacts.
     *
     * @param name  a contact name
     * @param phone the phone contact. May be {@code null}
     * @param fax   the fax contact. May be {@code null}
     * @param email the email contact. May be {@code null}
     * @return the corresponding {@code ContactType}
     */
    private ContactType getContact(String name, Contact phone, Contact fax, Contact email) {
        ContactType contact = new ContactType();
        if (!StringUtils.isEmpty(name)) {
            contact.setName(UBLHelper.initName(new NameType(), name));
        }
        contact.setTelephone(getPhone(phone));
        contact.setTelefax(getFax(fax));
        contact.setElectronicMail(getEmail(email));
        return contact;
    }

    /**
     * Returns an {@code TelephoneType} for a <em>contact.phoneNumber</em>.
     *
     * @param contact the phone contact. May be {@code null}
     * @return a new {@code TelephoneType} or {@code null} if {@code contact} is null or unpopulated
     */
    private TelephoneType getPhone(Contact contact) {
        String phone = (contact != null) ? contacts.getPhone(contact) : null;
        return (phone != null) ? UBLHelper.initText(new TelephoneType(), phone) : null;
    }

    /**
     * Returns an {@code TelefaxType} for a fax contact.
     *
     * @param contact the fax contact. May be {@code null}
     * @return a new {@code TelefaxType} or {@code null} if {@code contact} is null or unpopulated
     */
    private TelefaxType getFax(Contact contact) {
        String fax = (contact != null) ? contacts.getPhone(contact) : null;
        return (fax != null) ? UBLHelper.initText(new TelefaxType(), fax) : null;
    }

    /**
     * Returns an {@code ElectronicMailType} for a <em>contact.email</em>.
     *
     * @param contact the email contact. May be {@code null}
     * @return a new {@code ElectronicMailType} or {@code null} if {@code contact} is null or unpopulated
     */
    private ElectronicMailType getEmail(Contact contact) {
        String email = null;
        if (contact != null) {
            IMObjectBean bean = service.getBean(contact);
            email = StringUtils.trimToNull(bean.getString("emailAddress"));
        }
        return (email != null) ? UBLHelper.initText(new ElectronicMailType(), email) : null;
    }

    /**
     * Returns an {@code AddressType} for the supplied <em>contact.location</em>.
     *
     * @param contact the location contact
     * @return the corresponding {@code AddressType}
     */
    private AddressType getAddress(Contact contact) {
        IMObjectBean bean = service.getBean(contact);

        AddressType result = new AddressType();
        AddressLineType addressLineType = new AddressLineType();
        LineType line = UBLHelper.initText(new LineType(), bean.getString("address"));
        addressLineType.setLine(line);

        String city = lookupService.getName(contact, "suburb");
        CityNameType cityName = UBLHelper.initName(new CityNameType(), city);

        String state = lookupService.getName(contact, "state");
        CountrySubentityType stateName = UBLHelper.initText(new CountrySubentityType(), state);

        PostalZoneType postCode = UBLHelper.initText(new PostalZoneType(), bean.getString("postcode"));

        result.getAddressLine().add(addressLineType);
        result.setCityName(cityName);
        result.setCountrySubentity(stateName);
        result.setPostalZone(postCode);
        return result;
    }

    /**
     * Returns an {@code ItemIdentificationType} for the given identifier.
     *
     * @param id the identifier
     * @return a new {@code ItemIdentificationType}
     */
    private ItemIdentificationType getItemIdentification(long id) {
        ItemIdentificationType result = new ItemIdentificationType();
        result.setID(UBLHelper.createID(id));
        return result;
    }

    /**
     * Returns an {@code ItemIdentificationType} for the given identifier.
     *
     * @param id the identifier
     * @return a new {@code ItemIdentificationType}
     */
    private ItemIdentificationType getItemIdentification(String id) {
        ItemIdentificationType result = new ItemIdentificationType();
        result.setID(UBLHelper.createID(id));
        return result;
    }

    /**
     * Returns a {@code CopyIndicatorType} with the specified value.
     *
     * @param value the indicator value
     * @return a new {@code CopyIndicatorType}
     */
    private CopyIndicatorType getCopyIndicatorType(boolean value) {
        CopyIndicatorType result = new CopyIndicatorType();
        result.setValue(value);
        return result;
    }

    /**
     * Returns the currency associated with the practice.
     *
     * @return the currency code
     */
    private Currency getCurrency() {
        return UBLHelper.getCurrency(practiceRules, currencies, service);
    }

}
