/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */
package org.openvpms.esci.adapter.map.invoice;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.esci.adapter.map.UBLFinancialType;
import org.openvpms.esci.adapter.map.UBLType;
import org.openvpms.esci.adapter.util.ESCIAdapterException;
import org.openvpms.esci.ubl.common.aggregate.TaxCategoryType;
import org.openvpms.esci.ubl.common.aggregate.TaxSchemeType;
import org.openvpms.esci.ubl.common.basic.PercentType;
import org.openvpms.esci.ubl.common.basic.TaxTypeCodeType;

import java.math.BigDecimal;

/**
 * Wrapper for an UBL <tt>TaxCategoryType</tt>.
 *
 * @author <a href="mailto:support@openvpms.org">OpenVPMS Team</a>
 * @version $LastChangedDate: 2006-05-02 05:16:31Z $
 */
public class UBLTaxCategory extends UBLFinancialType {

    /**
     * The tax category.
     */
    private final TaxCategoryType category;

    /**
     * Constructs an <tt>UBLTaxCategory</tt>.
     *
     * @param category the tax category
     * @param parent   the parent
     * @param currency the expected currency for all amounts
     * @param service  the archetype service
     */
    public UBLTaxCategory(TaxCategoryType category, UBLType parent, String currency, IArchetypeService service) {
        super(parent, currency, service);
        this.category = category;
    }

    /**
     * Returns the type name.
     *
     * @return the type name
     */
    public String getType() {
        return "TaxCategory";
    }

    /**
     * Returns the type's identifier.
     *
     * @return the identifier. May be {@code null}
     */
    public String getID() {
        return getId(category.getID());
    }

    /**
     * Returns the tax rate, expressed as a percentage.
     *
     * @return the tax rate, or <tt>0.0</tt> if none is provided
     * @throws ESCIAdapterException if the tax rate is incorrectly specified
     */
    public BigDecimal getTaxRate() {
        PercentType percent = getRequired(category.getPercent(), "Percent");
        return getRequired(percent.getValue(), "Percent");
    }

    /**
     * Returns the tax scheme's tax type code.
     *
     * @return the tax scheme's tax type code
     * @throws ESCIAdapterException if no tax scheme is provided, or the tax scheme doesn't provide a tax type code
     */
    public String getTaxTypeCode() {
        TaxSchemeType scheme = getRequired(category.getTaxScheme(), "TaxScheme");
        TaxTypeCodeType result = getRequired(scheme.getTaxTypeCode(), "TaxScheme/TaxTypeCode");
        return getRequired(StringUtils.trimToNull(result.getValue()), "TaxScheme/TaxTypeCode");
    }

}
