/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.esci.adapter.map.invoice;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.product.ProductRules;
import org.openvpms.archetype.rules.product.ProductSupplier;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.component.service.lookup.LookupService;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Maps unit of measure codes to package units.
 *
 * @author Tim Anderson
 */
class PackageHelper {

    /**
     * Product rules.
     */
    private final ProductRules rules;

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The lookup service.
     */
    private final LookupService lookups;

    /**
     * Cache of lookup.uom codes to unit codes.
     */
    private Map<String, String> unitCodes;

    /**
     * Constructs a {@link PackageHelper}.
     *
     * @param rules   the product rules
     * @param service the archetype service
     * @param lookups the lookup service
     */
    public PackageHelper(ProductRules rules, ArchetypeService service, LookupService lookups) {
        this.rules = rules;
        this.lookups = lookups;
        this.service = service;
    }

    /**
     * Tries to determine the package information given the reorder code
     * (the {@link UBLInvoiceLine#getSellersItemID()}), the order item, product and supplier.
     * <p/>
     * In general, package information received from suppliers is not as accurate as the data maintained by the
     * practice, so use this where possible.<br/>
     * The package details are determined by the following, in order of availability:
     * <ol>
     *     <li>the reorder code, obtained from the supplier invoice line</li>
     *     <li>the reorder code on the order item</li>
     *     <li>the package size and units on the order item</li>
     *     <li>the product/supplier relationship, where there is a single relationship</li>
     * </ol>
     * For 1. and 2., the reorder code is used to look up the details on the corresponding product-supplier
     * relationship.<br/>
     * For 3. the product-supplier relationship with matching package size and units will be used, if present.<br/>
     *
     * @param reorderCode the reorder code. May be {@code null}
     * @param orderItem   the order item. May be {@code null}
     * @param product     the product. May be {@code null}
     * @param supplier    the supplier
     * @return the package information, or {@code null} if none is available
     */
    public Package getPackage(String reorderCode, FinancialAct orderItem, Product product, Party supplier) {
        Package result = null;
        List<ProductSupplier> productSuppliers = (product != null) ? rules.getProductSuppliers(product, supplier)
                                                                   : Collections.emptyList();
        if (reorderCode != null) {
            result = getPackage(reorderCode, productSuppliers);
        }
        if (result == null && orderItem != null) {
            // try and determine the package from the order item
            IMObjectBean bean = service.getBean(orderItem);
            String orderReorderCode = bean.getString("reorderCode");
            if (!StringUtils.isEmpty(orderReorderCode)) {
                result = getPackage(orderReorderCode, productSuppliers);
            }
            if (result == null) {
                int packageSize = bean.getInt("packageSize");
                String packageUnits = bean.getString("packageUnits");
                if (packageSize != 0 && !StringUtils.isEmpty(packageUnits)) {
                    // see if there is a matching product-supplier relationship
                    result = getPackage(packageSize, packageUnits, productSuppliers);
                    if (result == null) {
                        // no match, so use the order item details
                        result = new Package(packageSize, packageUnits);
                    }
                }
            }
        }
        if (result == null && productSuppliers.size() == 1) {
            // if no reorder code or order details were specified but there is a product/supplier, use that
            result = new Package(productSuppliers.get(0));
        }
        return result;
    }

    /**
     * Returns a package from a list of product-supplier relationships, based on reorder code.
     *
     * @param reorderCode      the reorder code
     * @param productSuppliers the product-supplier relationships
     * @return the matching package. May be {@code null}
     */
    private Package getPackage(String reorderCode, List<ProductSupplier> productSuppliers) {
        return productSuppliers.stream().filter(ps -> reorderCode.equals(ps.getReorderCode()))
                .map(Package::new)
                .findFirst()
                .orElse(null);
    }

    /**
     * Returns a package from a list of product-supplier relationships, based on package details.
     *
     * @param packageSize      the package size
     * @param packageUnits     the package units
     * @param productSuppliers the product-supplier relationships
     * @return the matching package. May be {@code null}
     */
    private Package getPackage(int packageSize, String packageUnits, List<ProductSupplier> productSuppliers) {
        return productSuppliers.stream().filter(ps -> packageSize == ps.getPackageSize()
                                                      && packageUnits.equals(ps.getPackageUnits()))
                .map(Package::new)
                .findFirst()
                .orElse(null);
    }

    /**
     * Returns the package units corresponding to a unit code.
     * <p/>
     * If there is more than one match, the resulting list will be sorted on package unit code.
     *
     * @param unitCode the unit code
     * @return the matching package units
     */
    public List<String> getPackageUnits(String unitCode) {
        List<String> result = new ArrayList<>();
        if (unitCodes == null) {
            unitCodes = getUnitCodes();
        }
        for (Map.Entry<String, String> entry : unitCodes.entrySet()) {
            if (StringUtils.equals(entry.getValue(), unitCode)) {
                result.add(entry.getKey());
            }
        }
        if (result.size() > 1) {
            Collections.sort(result);
        }
        return result;
    }

    /**
     * Returns a map of package unit codes to their corresponding UN/ECE codes.
     *
     * @return the map
     */
    private Map<String, String> getUnitCodes() {
        Map<String, String> result = new HashMap<>();
        for (Lookup lookup : lookups.getLookups("lookup.uom")) {
            IMObjectBean lookupBean = service.getBean(lookup);
            String unitCode = lookupBean.getString("unitCode");
            if (unitCode != null) {
                result.put(lookup.getCode(), unitCode);
            }
        }
        return result;
    }
}
