/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */
package org.openvpms.esci.adapter.dispatcher;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.message.MessageArchetypes;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.user.User;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Base class for listeners that create an <em>act.systemMessage</em> for the events they receive.
 *
 * @author Tim Anderson
 */
public abstract class AbstractSystemMessageFactory {

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * The logger.
     */
    private static final Logger log = LoggerFactory.getLogger(AbstractSystemMessageFactory.class);

    /**
     * Constructs an {@link AbstractSystemMessageFactory}.
     *
     * @param service the archetype service
     */
    protected AbstractSystemMessageFactory(ArchetypeService service) {
        this.service = service;
    }

    /**
     * Creates a system message linked to the supplied act and addressed to the act's author.
     * <p/>
     * The act's author is determined using {@link #getAddressee}. If there is no author, no system message will be
     * created.
     *
     * @param act     the act
     * @param subject the message subject
     * @param reason  the reason
     * @param message the message. May be {@code null}. Long messages are truncated
     * @return the system message, if one was created
     */
    protected Act createMessage(Act act, String subject, String reason, String message) {
        Act result = null;
        User user = getAddressee(act);
        if (user != null) {
            result = service.create(MessageArchetypes.SYSTEM, Act.class);
            IMObjectBean bean = service.getBean(result);
            bean.addTarget("item", act);
            bean.setTarget("to", user);
            bean.setValue("description", subject);
            bean.setValue("reason", reason);
            if (message != null) {
                message = StringUtils.abbreviate(message, bean.getMaxLength("message"));
                bean.setValue("message", message);
            }
            bean.save();
        } else {
            log.error("Cannot create system message for {}:{}, subject=\"{}\", reason=\"{}\" message=\"{}\". " +
                      "No user found to address the message to.", act.getArchetype(), act.getId(), subject, reason,
                      message);
        }
        return result;
    }

    /**
     * Returns the user to address a message to.
     * <p/>
     * By default, this is the createdBy user of the supplied act. If no createdBy user is present, and the act
     * has an associated stock location, the stock location's default author will be used, if any.
     *
     * @param act    the act
     * @return the author, or {@code null} if none is found
     */
    protected User getAddressee(Act act) {
        IMObjectBean bean = service.getBean(act);
        User result = null;
        if (act.getCreatedBy() != null) {
            result = service.get(act.getCreatedBy(), User.class);
        }
        if (result == null && bean.hasNode("stockLocation")) {
            Entity stockLocation = bean.getTarget("stockLocation", Entity.class);
            if (stockLocation != null) {
                IMObjectBean locBean = service.getBean(stockLocation);
                result = locBean.getTarget("defaultAuthor", User.class);
            }
        }
        return result;
    }

    /**
     * Returns the archetype service.
     *
     * @return the archetype service
     */
    protected ArchetypeService getService() {
        return service;
    }
}
