/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.esci.adapter.client.jaxws;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.i18n.Message;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.security.crypto.PasswordEncryptor;
import org.openvpms.esci.adapter.client.SupplierServiceLocator;
import org.openvpms.esci.adapter.dispatcher.ESCIConfig;
import org.openvpms.esci.adapter.i18n.ESCIAdapterMessages;
import org.openvpms.esci.adapter.util.ESCIAdapterException;
import org.openvpms.esci.service.InboxService;
import org.openvpms.esci.service.OrderService;
import org.openvpms.esci.service.RegistryService;
import org.openvpms.esci.service.client.ServiceLocator;
import org.openvpms.esci.service.client.ServiceLocatorFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.remoting.RemoteAccessException;
import org.springframework.remoting.jaxws.JaxWsSoapFaultException;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;

import javax.xml.ws.WebServiceException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Proxy;
import java.net.MalformedURLException;
import java.util.Collections;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;


/**
 * Returns proxies for supplier web services.
 * <p>
 * This supports timing out calls that take too long.
 *
 * @author Tim Anderson
 */
public class SupplierWebServiceLocator implements SupplierServiceLocator, DisposableBean {

    /**
     * The encryptor used to decrypt passwords.
     */
    private final PasswordEncryptor encryptor;

    /**
     * The service locator factory.
     */
    private final ServiceLocatorFactory locatorFactory;

    /**
     * The timeout for calls, in seconds or {@code 0} if calls shouldn't time out.
     */
    private final int timeout;

    /**
     * The executor service.
     */
    private final ExecutorService executor;

    /**
     * Constructs a {@link SupplierWebServiceLocator}.
     * <p>
     * Calls will time out after 30 seconds.
     *
     * @param encryptor the encryptor used to decrypt passwords
     * @param factory   the service locator factory
     */
    public SupplierWebServiceLocator(PasswordEncryptor encryptor, ServiceLocatorFactory factory) {
        this(encryptor, factory, 30);
    }

    /**
     * Constructs a {@link SupplierWebServiceLocator}.
     *
     * @param encryptor the encryptor used to decrypt passwords
     * @param factory   the service locator factory
     * @param timeout   the timeout for making calls to web services in seconds, or {@code 0} to not time out
     */
    public SupplierWebServiceLocator(PasswordEncryptor encryptor, ServiceLocatorFactory factory, int timeout) {
        this.encryptor = encryptor;
        this.locatorFactory = factory;
        this.timeout = timeout;
        executor = Executors.newCachedThreadPool(new CustomizableThreadFactory("ESCI-"));
    }

    /**
     * Returns a proxy for a supplier's {@link OrderService}.
     *
     * @param config the ESCI configuration
     * @return a proxy for the service provided by the supplier
     * @throws ESCIAdapterException if the associated {@code serviceURL} is invalid or the proxy cannot be created
     */
    @Override
    public OrderService getOrderService(ESCIConfig config) {
        SupplierServices services = new SupplierServices(config);
        return services.getOrderService();
    }

    /**
     * Returns a proxy for a supplier's {@link OrderService}.
     *
     * @param serviceURL the WSDL document URL of the service
     * @param username   the username to connect to the service with
     * @param password   the password to connect  to the service with
     * @return a proxy for the service provided by the supplier
     * @throws ESCIAdapterException if the associated {@code serviceURL} is invalid or the proxy cannot be created
     */
    public OrderService getOrderService(String serviceURL, String username, String password) {
        SupplierServices services = new SupplierServices(serviceURL, username, password);
        return services.getOrderService();
    }

    /**
     * Returns a proxy for the supplier's {@link InboxService}.
     *
     * @param config the ESCI configuration
     * @return a proxy for the service provided by the supplier
     * @throws ESCIAdapterException if the associated {@code serviceURL} is invalid or the proxy cannot be created
     */
    @Override
    public InboxService getInboxService(ESCIConfig config) {
        SupplierServices services = new SupplierServices(config);
        return services.getInboxService();
    }

    /**
     * Invoked by a BeanFactory on destruction of a singleton.
     *
     * @throws Exception in case of shutdown errors
     */
    @Override
    public void destroy() throws Exception {
        executor.shutdown();
    }

    /**
     * Helper class to access supplier services.
     */
    protected class SupplierServices {

        /**
         * The supplier.
         */
        private final Party supplier;

        /**
         * The registry service URL.
         */
        private final String serviceURL;

        /**
         * The user to connect as.
         */
        private final String username;

        /**
         * The user's password.
         */
        private final String password;


        /**
         * Constructs a {@code SupplierServices} from an <em>entityRelationship.supplierStockLocationESCI</em>
         * associated with the supplier and stock location.
         *
         * @param config the ESCI configuration
         * @throws ESCIAdapterException if there is no relationship or the URL is invalid
         */
        public SupplierServices(ESCIConfig config) {
            this.supplier = config.getSupplier();
            username = config.getUsername();
            String encrypted = config.getPassword();
            if (encrypted != null) {
                try {
                    password = encryptor.decrypt(encrypted);
                } catch (Exception exception) {
                    throw new ESCIAdapterException(ESCIAdapterMessages.failedToDecryptPassword(supplier, config.getStockLocation()));
                }
            } else {
                password = null;
            }

            serviceURL = config.getServiceURL();
            if (StringUtils.isEmpty(serviceURL)) {
                throw new ESCIAdapterException(ESCIAdapterMessages.invalidSupplierURL(supplier, serviceURL));
            }
        }

        /**
         * Constructs a {@link SupplierServices}.
         *
         * @param serviceURL the registry service URL.
         * @param username   the user name to connect as
         * @param password   the user's password
         */
        public SupplierServices(String serviceURL, String username, String password) {
            this.supplier = null;
            this.serviceURL = serviceURL;
            this.username = username;
            this.password = password;
        }

        /**
         * Returns the registry service proxy.
         *
         * @return the registry service proxy
         */
        public RegistryService getRegistryService() {
            return getRegistryService(null);
        }

        /**
         * Returns the registry service proxy.
         *
         * @param endpointAddress the endpoint address. May be {@code null}
         * @return the registry service proxy
         * @throws ESCIAdapterException for any error
         */
        public RegistryService getRegistryService(String endpointAddress) {
            return getService(RegistryService.class, serviceURL, endpointAddress);
        }

        /**
         * Returns the order service proxy.
         *
         * @return the order service proxy
         */
        public OrderService getOrderService() {
            return getOrderService(null, null);
        }

        /**
         * Returns the order service proxy.
         *
         * @param orderEndpointAddress    the order endpoint address. May be {@code null}
         * @param registryEndpointAddress the registry endpoint address. May be {@code null}
         * @return the order service proxy
         * @throws ESCIAdapterException for any error
         */
        public OrderService getOrderService(String orderEndpointAddress, String registryEndpointAddress) {
            RegistryService registry = getRegistryService(registryEndpointAddress);
            String order;
            try {
                order = registry.getOrderService();
            } catch (ESCIAdapterException exception) {
                throw exception;
            } catch (Exception exception) {
                throw new ESCIAdapterException(getConnectionFailed(), exception);
            }
            return getService(OrderService.class, order, orderEndpointAddress);
        }

        /**
         * Returns the inbox service proxy.
         *
         * @return the inbox service proxy
         * @throws ESCIAdapterException for any error
         */
        public InboxService getInboxService() {
            return getInboxService(null);
        }

        /**
         * Returns the inbox service proxy.
         *
         * @param endpointAddress the inbox service endpoint address. May be {@code null}
         * @return the inbox service proxy
         * @throws ESCIAdapterException for any error
         */
        public InboxService getInboxService(String endpointAddress) {
            RegistryService registry = getRegistryService();
            String inbox;
            try {
                inbox = registry.getInboxService();
            } catch (ESCIAdapterException exception) {
                throw exception;
            } catch (Exception exception) {
                throw new ESCIAdapterException(getConnectionFailed(), exception);
            }
            return getService(InboxService.class, inbox, endpointAddress);
        }

        /**
         * Proxies a web-service interface, forcing invocations to time out if they don't complete in time.
         *
         * @param service the service
         * @param type    the interface to proxy
         * @return the service proxy
         */
        @SuppressWarnings("unchecked")
        protected <T> T proxy(final T service, Class<T> type) {
            InvocationHandler handler = (proxy, method, args) -> {
                Callable<Object> callable = () -> method.invoke(service, args);
                Object result;
                try {
                    result = invokeWithTimeout(callable);
                } catch (ExecutionException exception) {
                    // unwrap the exception and rethrow as an ESCIAdapterException
                    Throwable cause = exception.getCause();
                    if (cause instanceof InvocationTargetException) {
                        cause = cause.getCause();
                    }
                    // the fallowing are thrown by JaxWsPortClientInterceptor
                    if (cause instanceof JaxWsSoapFaultException) {
                        cause = cause.getCause();
                        throw new ESCIAdapterException(ESCIAdapterMessages.remoteServiceError(cause.getMessage()), cause);
                    } else if (cause instanceof RemoteAccessException) {
                        cause = cause.getCause();
                        throw new ESCIAdapterException(getConnectionFailed(), cause);
                    }
                    throw cause;
                }
                return result;
            };
            return (T) Proxy.newProxyInstance(type.getClassLoader(), new Class[]{type}, handler);
        }

        /**
         * Returns a service proxy.
         * <p/>
         * This implementation supports returning proxies that can forcibly interrupt calls if they take too long.
         * <p/>
         * This is a workaround for network errors that prevent a service from responding.
         *
         * @param clazz           the proxy class
         * @param url             the service URL
         * @param endpointAddress the endpoint address. May be {@code null}
         * @return the service proxy
         * @throws ESCIAdapterException for any error
         */
        private <T> T getService(Class<T> clazz, String url, String endpointAddress) {
            T service;
            if (timeout > 0) {
                try {
                    service = invokeWithTimeout(() -> doGetService(clazz, url, endpointAddress));
                } catch (ExecutionException exception) {
                    Throwable cause = exception.getCause();
                    if (cause instanceof InvocationTargetException) {
                        cause = cause.getCause();
                    }
                    throw new ESCIAdapterException(getConnectionFailed(), cause);
                }
                service = proxy(service, clazz);
            } else {
                try {
                    service = doGetService(clazz, url, endpointAddress);
                } catch (ESCIAdapterException exception) {
                    throw exception;
                } catch (Exception exception) {
                    throw new ESCIAdapterException(getConnectionFailed(), exception);
                }
            }
            return service;
        }

        /**
         * Returns a service proxy.
         *
         * @param clazz           the proxy class
         * @param url             the service URL
         * @param endpointAddress the endpoint address. May be {@code null}
         * @return the service proxy
         * @throws ESCIAdapterException for any error
         */
        protected  <T> T doGetService(Class<T> clazz, String url, String endpointAddress) {
            ServiceLocator<T> locator;
            try {
                locator = locatorFactory.getServiceLocator(clazz, url, endpointAddress, username, password);
            } catch (MalformedURLException exception) {
                Message message = (supplier != null) ? ESCIAdapterMessages.invalidSupplierURL(supplier, serviceURL)
                                                     : ESCIAdapterMessages.invalidServiceURL(serviceURL);
                throw new ESCIAdapterException(message, exception);
            }
            try {
                return locator.getService();
            } catch (WebServiceException exception) {
                throw new ESCIAdapterException(getConnectionFailed(), exception);
            }
        }

        /**
         * Makes a web-service call, timing out if it doesn't complete in time.
         *
         * @param callable the call to make
         * @return the result of the call
         * @throws ESCIAdapterException for any error, including connection timeout
         * @throws ExecutionException   if execution fails
         */
        private <T> T invokeWithTimeout(Callable<T> callable) throws ExecutionException, ESCIAdapterException {
            T result;

            try {
                result = executor.invokeAny(Collections.singletonList(callable), timeout, TimeUnit.SECONDS);
            } catch (InterruptedException | TimeoutException exception) {
                if (exception instanceof InterruptedException) {
                    Thread.currentThread().interrupt();
                }
                if (supplier != null) {
                    throw new ESCIAdapterException(ESCIAdapterMessages.connectionTimedOut(supplier, serviceURL), exception);
                } else {
                    throw new ESCIAdapterException(ESCIAdapterMessages.connectionTimedOut(serviceURL), exception);
                }
            }
            return result;
        }

        /**
         * Helper to return a message for a connection failure.
         *
         * @return a new message
         */
        private Message getConnectionFailed() {
            return (supplier != null) ? ESCIAdapterMessages.connectionFailed(supplier, serviceURL)
                                      : ESCIAdapterMessages.connectionFailed(serviceURL);
        }

    }

}

