/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */
package org.openvpms.esci.adapter.map.order;

import org.openvpms.archetype.rules.supplier.OrderStatus;
import org.openvpms.component.i18n.Message;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.esci.adapter.i18n.ESCIAdapterMessages;
import org.openvpms.esci.adapter.map.AbstractUBLMapper;
import org.openvpms.esci.ubl.order.OrderResponseSimpleType;


/**
 * Maps UBL order responses to their corresponding orders.
 *
 * @author Tim Anderson
 */
public class OrderResponseMapperImpl extends AbstractUBLMapper implements OrderResponseMapper {

    /**
     * Constructs an {@link OrderResponseMapperImpl}.
     *
     * @param service the archetype service
     */
    public OrderResponseMapperImpl(ArchetypeService service) {
        super(service);
    }

    /**
     * Maps an {@link OrderResponseSimpleType} to its corresponding order.
     *
     * @param response      the response
     * @param supplier      the supplier that submitted the response
     * @param stockLocation the stock location
     * @param accountId     the supplier assigned account identifier. May be {@code null}
     * @return the corresponding order
     * @throws DuplicateOrderResponseException if the response is for an order that has already been ACCEPTED or
     * REJECTED
     */
    public FinancialAct map(OrderResponseSimpleType response, Party supplier, Party stockLocation, String accountId) {
        UBLOrderResponseSimple wrapper = new UBLOrderResponseSimple(response, getService());
        checkUBLVersion(wrapper);

        // check that the response isn't a duplicate
        FinancialAct order = wrapper.getOrder();
        checkDuplicateResponse(order, response);

        // check that the response is for the expected supplier and stock location
        wrapper.checkSupplier(supplier, accountId);
        wrapper.checkStockLocation(stockLocation, accountId);

        // check that the associated order is for the expected supplier and stock location
        checkOrder(order, supplier, stockLocation, wrapper);

        Message message;
        String status;
        if (wrapper.isAccepted()) {
            status = OrderStatus.ACCEPTED;
            message = ESCIAdapterMessages.orderAccepted();
        } else {
            status = OrderStatus.REJECTED;
            String note = wrapper.getRejectionNote();
            if (note != null) {
                message = ESCIAdapterMessages.orderRejected(note);
            } else {
                message = ESCIAdapterMessages.orderRejectedNoReason();
            }
        }
        IMObjectBean bean = getService().getBean(order);
        bean.setValue("status", status);
        bean.setValue("supplierResponse", message.getMessage());
        return order;
    }

    /**
     * Checks to see if an order response is a duplicate.
     *
     * @param order    the order
     * @param response the order response
     * @throws DuplicateOrderResponseException if the response is a duplicate
     */
    private void checkDuplicateResponse(FinancialAct order, OrderResponseSimpleType response) {
        String status = order.getStatus();
        if (OrderStatus.ACCEPTED.equals(status) || OrderStatus.REJECTED.equals(status)) {
            throw new DuplicateOrderResponseException(ESCIAdapterMessages.duplicateOrderResponse(order.getId(),
                                                                                                 response.getID().getValue()));
        }
    }

}
