/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.domain.internal.patient;

import org.apache.commons.lang3.StringUtils;
import org.junit.Test;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.lookup.TestLookupFactory;
import org.openvpms.domain.internal.service.patient.PatientBuilderImpl;
import org.openvpms.domain.internal.service.patient.PatientServices;
import org.openvpms.domain.patient.Patient;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Tests the {@link PatientBuilderImpl}.
 *
 * @author Tim Anderson
 */
public class PatientBuilderImplTestCase extends ArchetypeServiceTest {

    /**
     * The patient services.
     */
    @Autowired
    private PatientServices patientServices;

    /**
     * The lookup factory.
     */
    @Autowired
    private TestLookupFactory lookupFactory;

    /**
     * Verify that the description is correctly derived from the populated fields.
     */
    @Test
    public void testDescription() {
        lookupFactory.getBreed("KELPIE", "CANINE");
        PatientBuilderImpl builder = new PatientBuilderImpl(patientServices);
        Patient patient = builder.name("Fido")
                .species("CANINE")
                .breed("KELPIE")
                .sex(Patient.Sex.MALE)
                .colour("Black")
                .microchip("123466789")
                .build();
        assertEquals("Male Black Kelpie (Microchip: 123466789)", patient.getDescription());

        PatientBuilderImpl builder2 = new PatientBuilderImpl(patient, patientServices);
        builder2.microchip("987654321")
                .build();

        assertEquals("Male Black Kelpie (Microchip: 987654321)", patient.getDescription());

        // verifies that if the description exceeds 255 characters, it is abbreviated
        List<String> microchips = new ArrayList<>();
        for (int i = 0; i < 10; ++i) {
            String microchip = StringUtils.repeat("" + i, 20);
            builder2.addMicrochip(microchip);
            microchips.add("Microchip: " + microchip);
        }
        builder2.build();
        String unabbreviated = "Male Black Kelpie (" + StringUtils.join(microchips, ", ") + ")";
        assertTrue(unabbreviated.length() > 255);
        String abbreviated = StringUtils.abbreviate(unabbreviated, 255);
        assertEquals(255, abbreviated.length());
        assertTrue(abbreviated.endsWith("..."));
        assertEquals(abbreviated, patient.getDescription());
    }
}
