/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.domain.internal.builder;

import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityIdentity;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.domain.sync.Changes;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Builder for domain objects backed by {@link Entity} instances, that have a unique identity.
 *
 * @author Tim Anderson
 */
public abstract class EntityBuilder<D, B extends EntityBuilder<D, B>>
        extends DomainObjectBuilder<D, Entity, B> {

    /**
     * The supported entity identity archetype.
     */
    private final String supportedIdArchetype;

    /**
     * The transaction manager.
     */
    private final PlatformTransactionManager transactionManager;

    /**
     * The entity name.
     */
    private final NodeValue name = new NodeValue("name");

    /**
     * The entity description.
     */
    private final NodeValue description = new NodeValue("description");

    /**
     * Determines if the entity is active.
     */
    private final NodeValue active = new NodeValue("active");

    /**
     * Tracks changes.
     */
    private Changes<Entity> changes;

    /**
     * The entity id archetype.
     */
    private String entityIdArchetype;

    /**
     * The entity id.
     */
    private String entityId;

    /**
     * The entity id name.
     */
    private String entityIdName;

    /**
     * Constructs an {@link EntityBuilder}.
     *
     * @param archetype            the entity archetype
     * @param supportedIdArchetype the supported entity identity archetypes
     * @param domainClass          the domain class
     * @param service              the archetype service
     * @param transactionManager   the transaction manager
     * @param domainService        the domain object factory
     */
    public EntityBuilder(String archetype, String supportedIdArchetype, Class<D> domainClass, ArchetypeService service,
                         PlatformTransactionManager transactionManager, DomainService domainService) {
        super(archetype, Entity.class, domainClass, service, domainService);
        this.supportedIdArchetype = supportedIdArchetype;
        this.transactionManager = transactionManager;
    }

    /**
     * Track changes.
     *
     * @param changes the changes
     * @return this
     */
    public B changes(Changes<Entity> changes) {
        this.changes = changes;
        return getThis();
    }

    /**
     * Sets the entity id, used for identifying the entity.
     *
     * @param archetype the entity identity archetype. Must be an <em>entityIdentity.*</em>
     * @param id        the identifier
     * @return this
     */
    public B entityId(String archetype, String id) {
        return entityId(archetype, id, null);
    }

    /**
     * Sets the entity id, used for identifying the entity.
     *
     * @param archetype the entity identity archetype. Must be an <em>entityIdentity.*</em>
     * @param id        the identifier
     * @param name      the display name for the entity id
     * @return this
     */
    public B entityId(String archetype, String id, String name) {
        this.entityIdArchetype = archetype;
        this.entityId = id;
        this.entityIdName = name;
        return getThis();
    }

    /**
     * Sets the display name for the entity id.
     *
     * @param name the display name for the entity id
     * @return this
     */
    public B entityIdName(String name) {
        this.entityIdName = name;
        return getThis();
    }

    /**
     * Sets the entity name.
     *
     * @param name the entity name
     * @return this
     */
    public B name(String name) {
        return setValue(this.name, name);
    }

    /**
     * Sets the entity description.
     *
     * @param description the entity description
     * @return this
     */
    public B description(String description) {
        return setValue(this.description, description);
    }

    /**
     * Determines if the entity is active or not.
     *
     * @param active if {@code true}, the entity is active, otherwise it is inactive
     * @return this
     */
    public B active(boolean active) {
        return setValue(this.active, active);
    }

    /**
     * Builds the object.
     * <p/>
     * This implementation builds the object in a transaction.
     *
     * @param save if {@code true}, save the object, and any related objects
     * @return the object
     */
    @Override
    public D build(boolean save) {
        TransactionTemplate template = new TransactionTemplate(transactionManager);
        return template.execute(status -> super.build(save));
    }

    /**
     * Returns the object to build.
     * <p/>
     * This implementation delegates to {@link #getEntity(String, String)}, creating a new object if there
     * is no match.
     *
     * @param archetype the archetype
     * @return the object to build
     */
    @Override
    protected Entity getObject(String archetype) {
        if (entityIdArchetype == null) {
            throw new IllegalStateException("No identifier archetype provided");
        }
        if (!TypeHelper.matches(entityIdArchetype, supportedIdArchetype)) {
            throw new IllegalStateException("Expected entity identity archetype: " + supportedIdArchetype
                                            + " but got: " + entityIdArchetype);
        }
        if (entityId == null) {
            throw new IllegalStateException("No identifier provided");
        }
        Entity result = getEntity(entityIdArchetype, entityId);
        if (result == null) {
            result = create(archetype);
        }
        return result;
    }

    /**
     * Builds the object.
     *
     * @param state the build state
     * @return {@code true} if changes were made, otherwise {@code false}
     */
    @Override
    protected boolean build(State state) {
        Entity object = state.getObject();
        boolean changed = populate(object, state.getBean());
        if (changed) {
            state.addChanged(object);
        }
        if (changes != null) {
            if (object.isNew()) {
                changes.added(object);
            } else if (changed) {
                changes.updated(object);
            }
        }
        reset();
        return changed;
    }

    /**
     * Creates a new entity, adding the identity.
     *
     * @param archetype the archetype to create
     * @return a new entity
     */
    protected Entity create(String archetype) {
        Entity entity = create(archetype, Entity.class);
        entity.addIdentity(createIdentity(entityIdArchetype, entityId, entityIdName));
        return entity;
    }

    /**
     * Populates an entity.
     *
     * @param entity the entity to populate
     * @param bean   a bean wrapping the entity
     * @return {@code true} if the entity was updated
     */
    protected boolean populate(Entity entity, IMObjectBean bean) {
        boolean changed = name.update(bean);
        changed |= description.update(bean);
        changed |= active.update(bean);
        return changed;
    }

    /**
     * Returns the entity corresponding to the identifier.
     *
     * @param archetype the entity id archetype
     * @param id        the entity id
     * @return the entity, or {@code null} if none exists
     */
    protected abstract Entity getEntity(String archetype, String id);

    /**
     * Resets the builder.
     */
    protected void reset() {
        changes = null;
        entityIdArchetype = null;
        entityId = null;
        entityIdName = null;
        reset(name, description, active);
    }

    /**
     * Creates an identity for an entity.
     *
     * @param archetype the identity archetype
     * @param id        the identifier
     * @param name      the identity name. If {@code null}, the id will be used
     * @return a new identity
     */
    protected EntityIdentity createIdentity(String archetype, String id, String name) {
        EntityIdentity identity = create(archetype, EntityIdentity.class);
        identity.setIdentity(id);
        if (name != null) {
            identity.setName(name);
        } else {
            identity.setName(id);
        }
        return identity;
    }

    /**
     * Updates relationships.
     *
     * @param bean    the bean
     * @param node    the relationship node name
     * @param targets the relationship targets
     * @return {@code true} if relationships were changed, otherwise {@code false}
     */
    protected boolean updateRelationships(IMObjectBean bean, String node, List<? extends Entity> targets) {
        boolean changed = false;
        Set<Reference> targetRefs = new HashSet<>();
        for (Entity entity : targets) {
            targetRefs.add(entity.getObjectReference());
        }
        List<Reference> existing = bean.getTargetRefs(node);
        Set<Reference> toAdd = new HashSet<>(targetRefs);
        existing.forEach(toAdd::remove);

        Set<Reference> toRemove = new HashSet<>(existing);
        toRemove.removeAll(targetRefs);

        for (Reference reference : toAdd) {
            bean.addTarget(node, reference);
            changed = true;
        }

        for (Reference reference : toRemove) {
            bean.removeTarget(node, reference);
            changed = true;
        }
        return changed;
    }
}
