/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.domain.internal.patient.record;

import org.junit.Test;
import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.patient.TestPatientFactory;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.domain.internal.factory.DomainService;
import org.openvpms.domain.internal.patient.record.builder.VisitBuilderImpl;
import org.openvpms.domain.internal.service.patient.PatientServices;
import org.openvpms.domain.patient.Patient;
import org.openvpms.domain.patient.record.Visit;
import org.openvpms.domain.practice.Location;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * Tests the  {@link VisitBuilderImpl} class.
 *
 * @author Tim Anderson
 */
public class VisitBuilderImplTestCase extends ArchetypeServiceTest {

    /**
     * The domain service.
     */
    @Autowired
    private DomainService domainService;

    /**
     * The patient services.
     */
    @Autowired
    private PatientServices patientServices;

    /**
     * The patient factory.
     */
    @Autowired
    private TestPatientFactory patientFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * Tests creating new visits.
     */
    @Test
    public void testCreateVisit() {
        Patient patient1 = domainService.create(patientFactory.createPatient(), Patient.class);
        Patient patient2 = domainService.create(patientFactory.createPatient(), Patient.class);
        Location location1 = domainService.create(practiceFactory.createLocation(), Location.class);
        Date today = DateRules.getToday();
        Visit visit1 = new VisitBuilderImpl(patientServices).patient(patient1)
                .date(DateRules.toOffsetDateTime(today))
                .location(location1)
                .build();

        checkVisit(visit1, patient1, today, location1);

        // verify a visit can be created with no location
        Visit visit2 = new VisitBuilderImpl(patientServices).patient(patient2)
                .date(DateRules.toOffsetDateTime(today))
                .location(null)
                .build();
        checkVisit(visit2, patient2, today, null);
    }

    /**
     * Verifies the same visit is used for successive days at the same practice location.
     */
    @Test
    public void testSameVisitUsedForVisitOnSuccessiveDays() {
        Date today = DateRules.getToday();
        Date date1 = DateRules.getDate(today, -2, DateUnits.DAYS);
        Date date2 = DateRules.getYesterday();
        VisitBuilderImpl builder = new VisitBuilderImpl(patientServices);
        Patient patient = domainService.create(patientFactory.createPatient(), Patient.class);
        Location location1 = domainService.create(practiceFactory.createLocation(), Location.class);
        Visit visit1 = builder.patient(patient)
                .date(DateRules.toOffsetDateTime(date1))
                .location(location1)
                .build();

        Visit visit2 = builder.patient(patient)
                .date(DateRules.toOffsetDateTime(date2))
                .location(location1)
                .build();

        assertEquals(visit1, visit2);
        checkVisit(visit1, patient, date1, location1);
        checkVisit(visit2, patient, date1, location1);
    }

    /**
     * Verifies a new visit is created if a visit is created at location 1 on day1, but location2 on day 2.
     */
    @Test
    public void testDifferentVisitAtDifferentLocationsOnSuccessiveDays() {
        Date today = DateRules.getToday();
        Date date1 = DateRules.getDate(today, -2, DateUnits.DAYS);
        Date date2 = DateRules.getYesterday();
        VisitBuilderImpl builder = new VisitBuilderImpl(patientServices);
        Patient patient = domainService.create(patientFactory.createPatient(), Patient.class);
        Location location1 = domainService.create(practiceFactory.createLocation(), Location.class);
        Visit visit1 = builder.patient(patient)
                .date(DateRules.toOffsetDateTime(date1))
                .location(location1)
                .build();

        Location location2 = domainService.create(practiceFactory.createLocation(), Location.class);
        Visit visit2 = builder.patient(patient)
                .date(DateRules.toOffsetDateTime(date2))
                .location(location2)
                .build();

        assertNotEquals(visit1, visit2);
        checkVisit(visit1, patient, date1, location1);
        checkVisit(visit2, patient, date2, location2);
    }

    /**
     * Verifies a visit matches that expected.
     *
     * @param visit    the visit
     * @param patient  the expected patient
     * @param date     the expected date
     * @param location the expected practice locationas
     */
    private void checkVisit(Visit visit, Patient patient, Date date, Location location) {
        assertEquals(patient, visit.getPatient());
        assertEquals(date, visit.getActivityStartTime());
        assertEquals(location, visit.getLocation());
        assertEquals(ActStatus.IN_PROGRESS, visit.getStatus());
    }
}
