/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.db.migration.V2_2;

import org.flywaydb.core.api.migration.jdbc.BaseJdbcMigration;
import org.openvpms.db.util.ArchetypeDescriptorRemover;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Migration for OVPMS-2336 Change toolbox archetype --load to update rather than remove and load archetypes.
 * <p/>
 * This removes any legacy archetypes that are no longer in use.
 *
 * @author Tim Anderson
 */
public class V2_2_0_7__OVPMS_2336 extends BaseJdbcMigration {

    /**
     * The tables containing archetypes.
     */
    private static final String[] TABLES = {"action_type_descriptors",
                                            "acts",
                                            "act_identities",
                                            "act_relationships",
                                            "archetype_descriptors",
                                            "assertion_type_descriptors",
                                            "contacts",
                                            "documents",
                                            "entities",
                                            "entity_identities",
                                            "entity_links",
                                            "entity_relationships",
                                            "granted_authorities",
                                            "lookups",
                                            "lookup_links",
                                            "lookup_relationships",
                                            "participations",
                                            "product_prices",
                                            "security_roles"
    };

    /**
     * Standard archetypes generated on a clean database using:<br/>
     * {@code
     * SELECT CONCAT('"', SUBSTRING_INDEX(name, '.', 2), '", ') AS name FROM archetype_descriptors ORDER BY name
     * }
     */
    private static final String[] STANDARD_ARCHETYPES = {"act.auditMessage",
                                                         "act.bankDeposit",
                                                         "act.calendarBlock",
                                                         "act.calendarEvent",
                                                         "act.calendarEventSeries",
                                                         "act.customerAccountBadDebt",
                                                         "act.customerAccountChargesCounter",
                                                         "act.customerAccountChargesCredit",
                                                         "act.customerAccountChargesInvoice",
                                                         "act.customerAccountClosingBalance",
                                                         "act.customerAccountCounterItem",
                                                         "act.customerAccountCreditAdjust",
                                                         "act.customerAccountCreditItem",
                                                         "act.customerAccountDebitAdjust",
                                                         "act.customerAccountInitialBalance",
                                                         "act.customerAccountInvoiceItem",
                                                         "act.customerAccountOpeningBalance",
                                                         "act.customerAccountPayment",
                                                         "act.customerAccountPaymentCash",
                                                         "act.customerAccountPaymentCheque",
                                                         "act.customerAccountPaymentCredit",
                                                         "act.customerAccountPaymentDiscount",
                                                         "act.customerAccountPaymentEFT",
                                                         "act.customerAccountPaymentOther",
                                                         "act.customerAccountRefund",
                                                         "act.customerAccountRefundCash",
                                                         "act.customerAccountRefundCheque",
                                                         "act.customerAccountRefundCredit",
                                                         "act.customerAccountRefundDiscount",
                                                         "act.customerAccountRefundEFT",
                                                         "act.customerAccountRefundOther",
                                                         "act.customerAlert",
                                                         "act.customerAppointment",
                                                         "act.customerCommunicationEmail",
                                                         "act.customerCommunicationMail",
                                                         "act.customerCommunicationNote",
                                                         "act.customerCommunicationPhone",
                                                         "act.customerDocumentAttachment",
                                                         "act.customerDocumentAttachmentVersion",
                                                         "act.customerDocumentForm",
                                                         "act.customerDocumentLetter",
                                                         "act.customerDocumentLetterVersion",
                                                         "act.customerEstimation",
                                                         "act.customerEstimationItem",
                                                         "act.customerOrderItemPharmacy",
                                                         "act.customerOrderPharmacy",
                                                         "act.customerReturnInvestigation",
                                                         "act.customerReturnItemInvestigation",
                                                         "act.customerReturnItemPharmacy",
                                                         "act.customerReturnPharmacy",
                                                         "act.customerTask",
                                                         "act.documentLogo",
                                                         "act.documentTemplate",
                                                         "act.HL7Message",
                                                         "act.laboratoryOrder",
                                                         "act.patientAlert",
                                                         "act.patientClinicalAddendum",
                                                         "act.patientClinicalEvent",
                                                         "act.patientClinicalLink",
                                                         "act.patientClinicalNote",
                                                         "act.patientClinicalProblem",
                                                         "act.patientCustomerNote",
                                                         "act.patientDocumentAttachment",
                                                         "act.patientDocumentAttachmentVersion",
                                                         "act.patientDocumentForm",
                                                         "act.patientDocumentImage",
                                                         "act.patientDocumentImageVersion",
                                                         "act.patientDocumentLetter",
                                                         "act.patientDocumentLetterVersion",
                                                         "act.patientInsuranceClaim",
                                                         "act.patientInsuranceClaimAttachment",
                                                         "act.patientInsuranceClaimDeposit",
                                                         "act.patientInsuranceClaimItem",
                                                         "act.patientInsurancePolicy",
                                                         "act.patientInvestigation",
                                                         "act.patientInvestigationResultItem",
                                                         "act.patientInvestigationResults",
                                                         "act.patientInvestigationVersion",
                                                         "act.patientMedication",
                                                         "act.patientPrescription",
                                                         "act.patientReminder",
                                                         "act.patientReminderItemEmail",
                                                         "act.patientReminderItemExport",
                                                         "act.patientReminderItemList",
                                                         "act.patientReminderItemPrint",
                                                         "act.patientReminderItemSMS",
                                                         "act.patientWeight",
                                                         "act.rosterEvent",
                                                         "act.smsMessage",
                                                         "act.smsReply",
                                                         "act.stockAdjust",
                                                         "act.stockAdjustItem",
                                                         "act.stockTransfer",
                                                         "act.stockTransferItem",
                                                         "act.subscription",
                                                         "act.supplierAccountChargesCredit",
                                                         "act.supplierAccountChargesInvoice",
                                                         "act.supplierAccountCreditItem",
                                                         "act.supplierAccountInvoiceItem",
                                                         "act.supplierAccountPayment",
                                                         "act.supplierAccountPaymentCash",
                                                         "act.supplierAccountPaymentCheque",
                                                         "act.supplierAccountPaymentCredit",
                                                         "act.supplierAccountPaymentEFT",
                                                         "act.supplierAccountRefund",
                                                         "act.supplierAccountRefundCash",
                                                         "act.supplierAccountRefundCheque",
                                                         "act.supplierAccountRefundCredit",
                                                         "act.supplierAccountRefundEFT",
                                                         "act.supplierDelivery",
                                                         "act.supplierDeliveryItem",
                                                         "act.supplierDocumentAttachment",
                                                         "act.supplierDocumentAttachmentVersion",
                                                         "act.supplierDocumentForm",
                                                         "act.supplierDocumentLetter",
                                                         "act.supplierDocumentLetterVersion",
                                                         "act.supplierOrder",
                                                         "act.supplierOrderItem",
                                                         "act.supplierReturn",
                                                         "act.supplierReturnItem",
                                                         "act.systemMessage",
                                                         "act.tillBalance",
                                                         "act.tillBalanceAdjustment",
                                                         "act.userMessage",
                                                         "actIdentity.insuranceClaim",
                                                         "actIdentity.insurancePolicy",
                                                         "actIdentity.patientInvestigation",
                                                         "actIdentity.smartflowsheet",
                                                         "actIdentity.supplierInvoiceESCI",
                                                         "actRelationship.appointmentClinicalEvent",
                                                         "actRelationship.bankDepositItem",
                                                         "actRelationship.calendarEventSeries",
                                                         "actRelationship.customerAccountAllocation",
                                                         "actRelationship.customerAccountCounterItem",
                                                         "actRelationship.customerAccountCreditItem",
                                                         "actRelationship.customerAccountInvoiceItem",
                                                         "actRelationship.customerAccountPaymentItem",
                                                         "actRelationship.customerAccountRefundItem",
                                                         "actRelationship.customerAccountReversal",
                                                         "actRelationship.customerAppointmentTask",
                                                         "actRelationship.customerDocumentAttachmentVersion",
                                                         "actRelationship.customerDocumentLetterVersion",
                                                         "actRelationship.customerEstimateInvoice",
                                                         "actRelationship.customerEstimationItem",
                                                         "actRelationship.customerOrderItemPharmacy",
                                                         "actRelationship.customerReturnItemInvestigation",
                                                         "actRelationship.customerReturnItemPharmacy",
                                                         "actRelationship.insuranceClaimAttachment",
                                                         "actRelationship.insuranceClaimDeposit",
                                                         "actRelationship.insuranceClaimDocument",
                                                         "actRelationship.insuranceClaimInvoiceItem",
                                                         "actRelationship.insuranceClaimItem",
                                                         "actRelationship.insuranceClaimPolicy",
                                                         "actRelationship.invoiceItemAlert",
                                                         "actRelationship.invoiceItemDispensing",
                                                         "actRelationship.invoiceItemDocument",
                                                         "actRelationship.invoiceItemInvestigation",
                                                         "actRelationship.invoiceItemReminder",
                                                         "actRelationship.patientClinicalEventChargeItem",
                                                         "actRelationship.patientClinicalEventItem",
                                                         "actRelationship.patientClinicalItemAddendum",
                                                         "actRelationship.patientClinicalProblemItem",
                                                         "actRelationship.patientCustomerNote",
                                                         "actRelationship.patientDocument",
                                                         "actRelationship.patientDocumentAttachmentVersion",
                                                         "actRelationship.patientDocumentImageVersion",
                                                         "actRelationship.patientDocumentLetterVersion",
                                                         "actRelationship.patientInvestigationOrder",
                                                         "actRelationship.patientInvestigationResultItem",
                                                         "actRelationship.patientInvestigationResults",
                                                         "actRelationship.patientInvestigationVersion",
                                                         "actRelationship.patientPrescriptionMedication",
                                                         "actRelationship.patientReminderCommunication",
                                                         "actRelationship.patientReminderItem",
                                                         "actRelationship.smsMessageReply",
                                                         "actRelationship.stockAdjustItem",
                                                         "actRelationship.stockTransferItem",
                                                         "actRelationship.supplierAccountCreditItem",
                                                         "actRelationship.supplierAccountInvoiceItem",
                                                         "actRelationship.supplierAccountPaymentItem",
                                                         "actRelationship.supplierAccountRefundItem",
                                                         "actRelationship.supplierDeliveryInvoice",
                                                         "actRelationship.supplierDeliveryItem",
                                                         "actRelationship.supplierDeliveryOrder",
                                                         "actRelationship.supplierDeliveryOrderItem",
                                                         "actRelationship.supplierDocumentAttachmentVersion",
                                                         "actRelationship.supplierDocumentLetterVersion",
                                                         "actRelationship.supplierOrderItem",
                                                         "actRelationship.supplierReturnCredit",
                                                         "actRelationship.supplierReturnItem",
                                                         "actRelationship.supplierReturnOrderItem",
                                                         "actRelationship.systemMessageItem",
                                                         "actRelationship.tillBalanceItem",
                                                         "assertion.archetypeRange",
                                                         "assertion.archetypeRangeProperties",
                                                         "assertion.candidateChildren",
                                                         "assertion.expression",
                                                         "assertion.localLookup",
                                                         "assertion.localLookupEntryProperties",
                                                         "assertion.lookup",
                                                         "assertion.number",
                                                         "assertion.password",
                                                         "assertion.regularExpression",
                                                         "assertion.stringcase",
                                                         "assertion.uniqueEntityRelationship",
                                                         "contact.email",
                                                         "contact.location",
                                                         "contact.phoneNumber",
                                                         "contact.website",
                                                         "descriptor.actionType",
                                                         "descriptor.archetype",
                                                         "descriptor.assertion",
                                                         "descriptor.assertionProperty",
                                                         "descriptor.assertionType",
                                                         "descriptor.collectionNode",
                                                         "descriptor.node",
                                                         "descriptor.propertyList",
                                                         "descriptor.propertyMap",
                                                         "descriptor.propertyType",
                                                         "document.HL7",
                                                         "document.image",
                                                         "document.other",
                                                         "document.temporary",
                                                         "document.text",
                                                         "entity.appointmentType",
                                                         "entity.cageType",
                                                         "entity.calendarBlockType",
                                                         "entity.calendarServiceRatio",
                                                         "entity.customPatientExample",
                                                         "entity.discountGroupType",
                                                         "entity.discountType",
                                                         "entity.documentTemplate",
                                                         "entity.documentTemplateEmailSystem",
                                                         "entity.documentTemplateEmailUser",
                                                         "entity.documentTemplateSMSAppointment",
                                                         "entity.documentTemplateSMSReminder",
                                                         "entity.entityMappings",
                                                         "entity.HL7Mapping",
                                                         "entity.HL7MappingCubex",
                                                         "entity.HL7MappingIDEXX",
                                                         "entity.HL7ReceiverMLLP",
                                                         "entity.HL7SenderMLLP",
                                                         "entity.HL7ServiceLaboratory",
                                                         "entity.HL7ServiceLaboratoryGroup",
                                                         "entity.HL7ServicePatientEvent",
                                                         "entity.HL7ServicePharmacy",
                                                         "entity.HL7ServicePharmacyGroup",
                                                         "entity.investigationType",
                                                         "entity.jobAppointmentReminder",
                                                         "entity.jobDocumentLoader",
                                                         "entity.jobESCIInboxReader",
                                                         "entity.jobMedicalRecordLocker",
                                                         "entity.jobPatientReminderQueue",
                                                         "entity.jobPatientReminderSender",
                                                         "entity.jobPharmacyOrderDiscontinuation",
                                                         "entity.jobScheduledReport",
                                                         "entity.laboratoryDevice",
                                                         "entity.laboratoryTest",
                                                         "entity.laboratoryTestHL7",
                                                         "entity.letterhead",
                                                         "entity.mailServer",
                                                         "entity.onlineBookingTimesType",
                                                         "entity.organisationScheduleView",
                                                         "entity.organisationWorkListView",
                                                         "entity.patientAlertType",
                                                         "entity.pluginConfiguration",
                                                         "entity.preferenceGroupCharge",
                                                         "entity.preferenceGroupGeneral",
                                                         "entity.preferenceGroupHistory",
                                                         "entity.preferenceGroupScheduling",
                                                         "entity.preferenceGroupSummary",
                                                         "entity.preferenceGroupWorkList",
                                                         "entity.preferences",
                                                         "entity.printer",
                                                         "entity.productBatch",
                                                         "entity.productDose",
                                                         "entity.productType",
                                                         "entity.reminderConfigurationType",
                                                         "entity.reminderCount",
                                                         "entity.reminderRule",
                                                         "entity.reminderType",
                                                         "entity.rosterArea",
                                                         "entity.smartflowConfigurationType",
                                                         "entity.SMSConfigEmailClickatell",
                                                         "entity.SMSConfigEmailGeneric",
                                                         "entity.SMSConfigEmailSMSGlobal",
                                                         "entity.taskType",
                                                         "entity.userGroup",
                                                         "entityIdentity.alias",
                                                         "entityIdentity.barcode",
                                                         "entityIdentity.code",
                                                         "entityIdentity.laboratoryTest",
                                                         "entityIdentity.microchip",
                                                         "entityIdentity.petTag",
                                                         "entityIdentity.rabiesTag",
                                                         "entityLink.batchLocation",
                                                         "entityLink.batchManufacturer",
                                                         "entityLink.batchProduct",
                                                         "entityLink.cageFirstPetProductDay",
                                                         "entityLink.cageFirstPetProductNight",
                                                         "entityLink.cageLateCheckoutProduct",
                                                         "entityLink.cageSecondPetProductDay",
                                                         "entityLink.cageSecondPetProductNight",
                                                         "entityLink.customerDiscount",
                                                         "entityLink.customerLocation",
                                                         "entityLink.discountType",
                                                         "entityLink.documentTemplateEmail",
                                                         "entityLink.documentTemplateSMS",
                                                         "entityLink.entityMapping",
                                                         "entityLink.followupWorkList",
                                                         "entityLink.HL7LaboratoryGroupLaboratory",
                                                         "entityLink.HL7Mapping",
                                                         "entityLink.HL7PharmacyGroupPharmacy",
                                                         "entityLink.HL7ServiceLocation",
                                                         "entityLink.HL7ServiceReceiver",
                                                         "entityLink.HL7ServiceSender",
                                                         "entityLink.HL7ServiceUser",
                                                         "entityLink.insuranceService",
                                                         "entityLink.insurerDocumentTemplate",
                                                         "entityLink.investigationTypeDevice",
                                                         "entityLink.investigationTypeLaboratory",
                                                         "entityLink.jobScheduledReportLocation",
                                                         "entityLink.jobScheduledReportTemplate",
                                                         "entityLink.jobUser",
                                                         "entityLink.jobUserGroup",
                                                         "entityLink.laboratoryDeviceLaboratory",
                                                         "entityLink.laboratoryDeviceLocation",
                                                         "entityLink.laboratoryLocation",
                                                         "entityLink.laboratoryTestInvestigationType",
                                                         "entityLink.letterheadContacts",
                                                         "entityLink.locationGapBenefitTill",
                                                         "entityLink.locationLetterhead",
                                                         "entityLink.locationPrinter",
                                                         "entityLink.locationProductType",
                                                         "entityLink.organisationMailServer",
                                                         "entityLink.organisationSMSAppointment",
                                                         "entityLink.patientDiscount",
                                                         "entityLink.pluginMapping",
                                                         "entityLink.practiceReminderConfiguration",
                                                         "entityLink.practiceServiceUser",
                                                         "entityLink.preferenceGroupCharge",
                                                         "entityLink.preferenceGroupGeneral",
                                                         "entityLink.preferenceGroupHistory",
                                                         "entityLink.preferenceGroupScheduling",
                                                         "entityLink.preferenceGroupSummary",
                                                         "entityLink.preferenceGroupWorkList",
                                                         "entityLink.preferencesUser",
                                                         "entityLink.productAlert",
                                                         "entityLink.productDiscount",
                                                         "entityLink.productDocument",
                                                         "entityLink.productDose",
                                                         "entityLink.productEquivalent",
                                                         "entityLink.productIncludes",
                                                         "entityLink.productLaboratoryTest",
                                                         "entityLink.productLink",
                                                         "entityLink.productLocation",
                                                         "entityLink.productPharmacy",
                                                         "entityLink.productReminder",
                                                         "entityLink.productStockLocation",
                                                         "entityLink.productSupplier",
                                                         "entityLink.productType",
                                                         "entityLink.productTypeDiscount",
                                                         "entityLink.reminderConfigurationLocation",
                                                         "entityLink.reminderConfigurationTemplateCustomer",
                                                         "entityLink.reminderConfigurationTemplatePatient",
                                                         "entityLink.reminderCountRule",
                                                         "entityLink.reminderCountTemplate",
                                                         "entityLink.reminderTypeCount",
                                                         "entityLink.rosterAreaLocation",
                                                         "entityLink.rosterAreaSchedule",
                                                         "entityLink.scheduleBookingTimes",
                                                         "entityLink.scheduleCageType",
                                                         "entityLink.scheduleLocation",
                                                         "entityLink.smartflowConfiguration",
                                                         "entityLink.supplierLocation",
                                                         "entityLink.userLocation",
                                                         "entityRelationship.customPatientFields",
                                                         "entityRelationship.documentTemplatePrinter",
                                                         "entityRelationship.investigationTypeSupplier",
                                                         "entityRelationship.investigationTypeTemplate",
                                                         "entityRelationship.locationDeposit",
                                                         "entityRelationship.locationOTC",
                                                         "entityRelationship.locationStockLocation",
                                                         "entityRelationship.locationTill",
                                                         "entityRelationship.locationView",
                                                         "entityRelationship.locationWorkListView",
                                                         "entityRelationship.patientLocation",
                                                         "entityRelationship.patientOwner",
                                                         "entityRelationship.practiceLocation",
                                                         "entityRelationship.practiceSMSConfiguration",
                                                         "entityRelationship.practiceVeterinarians",
                                                         "entityRelationship.referredFrom",
                                                         "entityRelationship.referredTo",
                                                         "entityRelationship.scheduleAppointmentType",
                                                         "entityRelationship.scheduleDocumentTemplate",
                                                         "entityRelationship.scheduleWorkList",
                                                         "entityRelationship.stockLocationDefaultAuthor",
                                                         "entityRelationship.supplierRepresentatives",
                                                         "entityRelationship.supplierStockLocationESCI",
                                                         "entityRelationship.userGroup",
                                                         "entityRelationship.viewSchedule",
                                                         "entityRelationship.viewWorkList",
                                                         "entityRelationship.worklistDocumentTemplate",
                                                         "entityRelationship.worklistTaskType",
                                                         "lookup.addressformat",
                                                         "lookup.bank",
                                                         "lookup.breed",
                                                         "lookup.colour",
                                                         "lookup.contactPurpose",
                                                         "lookup.country",
                                                         "lookup.creditcard",
                                                         "lookup.currency",
                                                         "lookup.customerAccountType",
                                                         "lookup.customerAlertType",
                                                         "lookup.customerCommunicationReason",
                                                         "lookup.customerInsurance",
                                                         "lookup.customerReferral",
                                                         "lookup.customerType",
                                                         "lookup.customerVet",
                                                         "lookup.customPaymentType",
                                                         "lookup.demographicUpdate",
                                                         "lookup.diagnosis",
                                                         "lookup.diagnosisVeNom",
                                                         "lookup.durationformat",
                                                         "lookup.durationformats",
                                                         "lookup.fileNameFormat",
                                                         "lookup.lookupMappings",
                                                         "lookup.macro",
                                                         "lookup.macroReport",
                                                         "lookup.messageReason",
                                                         "lookup.messageStatus",
                                                         "lookup.patientAlertType",
                                                         "lookup.personTitle",
                                                         "lookup.practiceType",
                                                         "lookup.presentingComplaint",
                                                         "lookup.presentingComplaintVeNom",
                                                         "lookup.pricingGroup",
                                                         "lookup.productDrugSchedule",
                                                         "lookup.productGroup",
                                                         "lookup.productIncomeType",
                                                         "lookup.reminderGroup",
                                                         "lookup.reportType",
                                                         "lookup.species",
                                                         "lookup.speciesIDEXX",
                                                         "lookup.state",
                                                         "lookup.suburb",
                                                         "lookup.supplierAccountType",
                                                         "lookup.supplierType",
                                                         "lookup.taxType",
                                                         "lookup.uom",
                                                         "lookup.uomGroup",
                                                         "lookup.userNameFormat",
                                                         "lookup.userType",
                                                         "lookup.veterinarySpeciality",
                                                         "lookup.visitReason",
                                                         "lookup.visitReasonVeNom",
                                                         "lookupLink.lookupMapping",
                                                         "lookupRelationship.breedMapping",
                                                         "lookupRelationship.countryState",
                                                         "lookupRelationship.customerAccountAlert",
                                                         "lookupRelationship.durationformats",
                                                         "lookupRelationship.speciesBreed",
                                                         "lookupRelationship.speciesMappingIDEXX",
                                                         "lookupRelationship.stateSuburb",
                                                         "lookupRelationship.uomGroup",
                                                         "participation.appointmentType",
                                                         "participation.calendarBlockType",
                                                         "participation.clinician",
                                                         "participation.customer",
                                                         "participation.customerAccountBalance",
                                                         "participation.deposit",
                                                         "participation.document",
                                                         "participation.documentTemplate",
                                                         "participation.HL7Connector",
                                                         "participation.insurer",
                                                         "participation.investigationType",
                                                         "participation.laboratory",
                                                         "participation.laboratoryDevice",
                                                         "participation.laboratoryTest",
                                                         "participation.location",
                                                         "participation.logo",
                                                         "participation.manufacturer",
                                                         "participation.medication",
                                                         "participation.patient",
                                                         "participation.patientAlertType",
                                                         "participation.product",
                                                         "participation.productBatch",
                                                         "participation.productTemplate",
                                                         "participation.reminderType",
                                                         "participation.rosterArea",
                                                         "participation.schedule",
                                                         "participation.smsRecipient",
                                                         "participation.stock",
                                                         "participation.stockLocation",
                                                         "participation.stockTransferLocation",
                                                         "participation.subscription",
                                                         "participation.supplier",
                                                         "participation.taskType",
                                                         "participation.till",
                                                         "participation.user",
                                                         "participation.worklist",
                                                         "party.customerperson",
                                                         "party.organisationDeposit",
                                                         "party.organisationLocation",
                                                         "party.organisationOTC",
                                                         "party.organisationPractice",
                                                         "party.organisationSchedule",
                                                         "party.organisationStockLocation",
                                                         "party.organisationTill",
                                                         "party.organisationWorkList",
                                                         "party.patientpet",
                                                         "party.supplierInsurer",
                                                         "party.supplierManufacturer",
                                                         "party.supplierorganisation",
                                                         "party.supplierperson",
                                                         "party.supplierVeterinarian",
                                                         "party.supplierVeterinaryPractice",
                                                         "product.medication",
                                                         "product.merchandise",
                                                         "product.priceTemplate",
                                                         "product.service",
                                                         "product.template",
                                                         "productPrice.fixedPrice",
                                                         "productPrice.unitPrice",
                                                         "security.archetypeAuthority",
                                                         "security.role",
                                                         "security.user"};

    /**
     * Executes this migration. The execution will automatically take place within a transaction, when the underlying
     * database supports it.
     *
     * @param connection The connection to use to execute statements.
     * @throws Exception when the migration failed.
     */
    @Override
    public void migrate(Connection connection) throws Exception {
        ArchetypeDescriptorRemover remover = new ArchetypeDescriptorRemover();

        // collect archetypes that are in use.
        Set<String> archetypes = getArchetypesInUse(connection);

        // collect archetype descriptors
        Map<String, Long> descriptors = remover.getArchetypes(connection);

        // exclude archetypes that included in the standard distribution
        for (String archetype : STANDARD_ARCHETYPES) {
            descriptors.remove(archetype);
        }

        // exclude archetypes that are in use
        for (String archetype : archetypes) {
            descriptors.remove(archetype);
        }

        // remove archetypes that remain
        for (long id : descriptors.values()) {
            remover.remove(id, connection);
        }
    }

    /**
     * Returns archetypes that are in use.
     *
     * @param connection the database connection
     * @return the archetypes that are in use
     * @throws SQLException for any SQL error
     */
    private Set<String> getArchetypesInUse(Connection connection) throws SQLException {
        Set<String> archetypes = new HashSet<>();
        for (String table : TABLES) {
            String sql = "SELECT DISTINCT arch_short_name FROM " + table;
            try (PreparedStatement statement = connection.prepareStatement(sql)) {
                try (ResultSet set = statement.executeQuery()) {
                    while (set.next()) {
                        String archetype = set.getString(1);
                        archetypes.add(archetype);
                    }
                }
            }
        }
        return archetypes;
    }

}