/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2022 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.billing.internal.charge;

import org.openvpms.archetype.rules.patient.MedicalRecordRules;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.user.User;
import org.openvpms.domain.patient.Patient;
import org.openvpms.domain.product.Template;

import java.util.Date;

/**
 * Adds a note from a {@link Template} to a patient history.
 *
 * @author Tim Anderson
 */
class TemplateVisitNoteBuilder {

    /**
     * The patient.
     */
    private final Patient patient;

    /**
     * The template.
     */
    private final Template template;

    /**
     * The clinician.
     */
    private final User clinician;

    /**
     * The visit note.
     */
    private final String note;

    /**
     * The builder service.
     */
    private final BuilderServices builderServices;

    /**
     * Constructs a {@link TemplateVisitNoteBuilder}.
     *
     * @param patient         the patient
     * @param template        the template
     * @param note            the note
     * @param clinician       the clinician
     * @param builderServices the builder services
     */
    public TemplateVisitNoteBuilder(Patient patient, Template template, String note, User clinician, BuilderServices builderServices) {
        this.patient = patient;
        this.template = template;
        this.note = note;
        this.clinician = clinician;
        this.builderServices = builderServices;
    }

    /**
     * Determines if this builder has the same patient and template.
     *
     * @param patient  the patient
     * @param template the template
     * @return {@code true} if they match, otherwise {@code false}
     */
    public boolean matches(Patient patient, Template template) {
        return this.patient.equals(patient) && this.template.equals(template);
    }

    /**
     * Builders the note record, adding it to the patient visit.
     * <p/>
     * The record will be linked to the first invoice item invoice items generated by the template.
     *
     * @param date    the date used to select the visit
     * @param context the invoice context
     */
    public void build(Date date, InvoiceContext context) {
        MedicalRecordRules rules = builderServices.getMedicalRecordRules();
        Act record = rules.createNote(date, patient, note, clinician);
        context.addRecord(record, date);
    }
}
