/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.workflow;

import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.helper.LookupHelper;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.Participation;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.lookup.LookupService;
import org.openvpms.component.system.common.util.PropertySet;

import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * A factory for appointments.
 *
 * @author Tim Anderson
 */
class AppointmentFactory extends ScheduleEventFactory {

    /**
     * Cache of visit reason lookup names, keyed on code.
     */
    private final Map<String, String> reasonNames = Collections.synchronizedMap(new HashMap<>());

    /**
     * Constructs an {@link AppointmentFactory}.
     *
     * @param service the archetype service
     * @param lookups the lookup service
     */
    public AppointmentFactory(IArchetypeService service, LookupService lookups) {
        super(ScheduleArchetypes.APPOINTMENT, service, lookups);
        Map<String, String> map = LookupHelper.getNames(service, lookups, ScheduleArchetypes.APPOINTMENT, "reason");
        reasonNames.putAll(map);
    }

    /**
     * Caches a visit reason.
     *
     * @param reason the reason to cache
     * @return {@code true} if the reason was already cached
     */
    public boolean addReason(Lookup reason) {
        return reasonNames.put(reason.getCode(), reason.getName()) != null;
    }

    /**
     * Removes a cached visit reason.
     *
     * @param reason the reason to remove
     * @return {@code true} if the reason was removed
     */
    public boolean removeReason(Lookup reason) {
        return reasonNames.remove(reason.getCode()) != null;
    }

    /**
     * Assembles an {@link PropertySet PropertySet} from a source act.
     *
     * @param target the target set
     * @param source the source act
     */
    @Override
    protected void assemble(PropertySet target, IMObjectBean source) {
        super.assemble(target, source);

        populate(target, source, "schedule");
        Act object = source.getObject(Act.class);

        if (source.isA(ScheduleArchetypes.APPOINTMENT)) {
            String reason = object.getReason();
            target.set(ScheduleEvent.ACT_REASON, reason);
            target.set(ScheduleEvent.ACT_REASON_NAME, reasonNames.get(reason));

            Reference typeRef = source.getTargetRef("appointmentType");
            String typeName = getName(typeRef);
            target.set(ScheduleEvent.SCHEDULE_TYPE_REFERENCE, typeRef);
            target.set(ScheduleEvent.SCHEDULE_TYPE_NAME, typeName);
            target.set(ScheduleEvent.SEND_REMINDER, source.getBoolean(ScheduleEvent.SEND_REMINDER));
            target.set(ScheduleEvent.REMINDER_SENT, source.getDate(ScheduleEvent.REMINDER_SENT));
            target.set(ScheduleEvent.SMS_STATUS, source.getString(ScheduleEvent.SMS_STATUS));
            target.set(ScheduleEvent.REMINDER_ERROR, source.getString(ScheduleEvent.REMINDER_ERROR));
            target.set(ScheduleEvent.CONFIRMED_TIME, source.getDate(ScheduleEvent.CONFIRMED_TIME));
            target.set(ScheduleEvent.ARRIVAL_TIME, source.getDate(ScheduleEvent.ARRIVAL_TIME));
            target.set(ScheduleEvent.ONLINE_BOOKING, source.getBoolean(ScheduleEvent.ONLINE_BOOKING));
            target.set(ScheduleEvent.BOOKING_NOTES, source.getString(ScheduleEvent.BOOKING_NOTES));
        } else {
            target.set(ScheduleEvent.ACT_NAME, object.getName());
            Participation type = source.getObject("type", Participation.class);
            populate(target, type, "scheduleType");
        }
    }

    /**
     * Creates a query to query events for a particular entity between two times.
     *
     * @param entity    the entity
     * @param startTime the start time, inclusive
     * @param endTime   the end time, exclusive
     * @return a new query
     */
    @Override
    protected ScheduleEventQuery createQuery(Entity entity, Date startTime, Date endTime) {
        return new AppointmentQuery(entity, startTime, endTime, getStatusNames(), reasonNames, getService());
    }
}
