/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.lookup;

import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.math.BigDecimal;
import java.util.Set;

/**
 * Builder for <em>lookup.taxType</em> lookups, for testing purposes.
 *
 * @author Tim Anderson
 */
public class TestTaxTypeBuilder extends AbstractTestLookupBuilder<TestTaxTypeBuilder> {

    /**
     * The tax rate.
     */
    private ValueStrategy rate = ValueStrategy.unset();

    /**
     * Sets the tax scheme.
     */
    private ValueStrategy taxScheme = ValueStrategy.unset();

    /**
     * Sets the tax category.
     */
    private ValueStrategy taxCategory = ValueStrategy.unset();

    /**
     * Constructs a {@link TestTaxTypeBuilder}.
     *
     * @param service the archetype service
     */
    public TestTaxTypeBuilder(ArchetypeService service) {
        super("lookup.taxType", service);
        code(ValueStrategy.random("ZTAXTYPE").toString());
    }

    /**
     * Sets the tax rate.
     *
     * @param rate the tax rate
     * @return this
     */
    public TestTaxTypeBuilder rate(int rate) {
        return rate(BigDecimal.valueOf(rate));
    }

    /**
     * Sets the tax rate.
     *
     * @param rate the tax rate
     * @return this
     */
    public TestTaxTypeBuilder rate(BigDecimal rate) {
        this.rate = ValueStrategy.value(rate);
        return this;
    }

    /**
     * Sets the tax scheme.
     *
     * @param taxScheme the tax scheme
     * @return the tax scheme
     */
    public TestTaxTypeBuilder taxScheme(String taxScheme) {
        this.taxScheme = ValueStrategy.value(taxScheme);
        return this;
    }

    /**
     * Sets the tax category.
     *
     * @param taxCategory the tax category
     * @return this
     */
    public TestTaxTypeBuilder taxCategory(String taxCategory) {
        this.taxCategory = ValueStrategy.value(taxCategory);
        return this;
    }

    /**
     * Builds the object.
     *
     * @param object   the object
     * @param bean     a bean wrapping the object
     * @param toSave   objects to save, if the object is to be saved
     * @param toRemove the objects to remove
     */
    @Override
    protected void build(Lookup object, IMObjectBean bean, Set<IMObject> toSave, Set<Reference> toRemove) {
        super.build(object, bean, toSave, toRemove);
        rate.setValue(bean, "rate");
        taxScheme.setValue(bean, "taxScheme");
        taxCategory.setValue(bean, "taxCategory");
    }
}
