/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test;

import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.lang3.StringUtils;
import org.openvpms.archetype.rules.customer.CustomerArchetypes;
import org.openvpms.archetype.rules.finance.till.TillArchetypes;
import org.openvpms.archetype.rules.party.ContactArchetypes;
import org.openvpms.archetype.rules.party.PartyRules;
import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.archetype.rules.patient.PatientRules;
import org.openvpms.archetype.rules.practice.PracticeArchetypes;
import org.openvpms.archetype.rules.product.ProductArchetypes;
import org.openvpms.archetype.rules.supplier.SupplierArchetypes;
import org.openvpms.archetype.rules.user.UserArchetypes;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.business.service.archetype.ArchetypeServiceHelper;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.ValidationException;
import org.openvpms.component.business.service.archetype.helper.IMObjectBean;
import org.openvpms.component.business.service.archetype.helper.LookupHelper;
import org.openvpms.component.business.service.lookup.LookupServiceHelper;
import org.openvpms.component.model.act.ActIdentity;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityIdentity;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.lookup.LookupRelationship;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Relationship;
import org.openvpms.component.model.party.Contact;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.model.user.User;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.NodeConstraint;
import org.openvpms.component.system.common.query.QueryIterator;

import java.math.BigDecimal;
import java.security.SecureRandom;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Random;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.fail;


/**
 * Unit test helper.
 *
 * @author Tim Anderson
 */
public class TestHelper {

    /**
     * Random no. generator for creating unique names.
     */
    private static final Random random = new SecureRandom();

    /**
     * Creates a new object.
     *
     * @param shortName the archetype short name
     * @return a new object
     */
    public static IMObject create(String shortName) {
        IArchetypeService service = ArchetypeServiceHelper.getArchetypeService();
        IMObject object = service.create(shortName);
        assertNotNull(object);
        return object;
    }

    /**
     * Creates a new object.
     *
     * @param archetype the archetype name
     * @param type      the expected type of the object
     * @return a new object
     */
    public static <T extends IMObject> T create(String archetype, Class<T> type) {
        IArchetypeService service = ArchetypeServiceHelper.getArchetypeService();
        return service.create(archetype, type);
    }

    /**
     * Helper to save an object.
     *
     * @param object the object to save
     * @throws ArchetypeServiceException if the service cannot save the object
     * @throws ValidationException       if the object cannot be validated
     */
    public static void save(IMObject object) {
        ArchetypeServiceHelper.getArchetypeService().save(object);
    }

    /**
     * Helper to save an array of objects.
     *
     * @param objects the objects to save
     * @throws ArchetypeServiceException if the service cannot save the object
     * @throws ValidationException       if the object cannot be validated
     */
    public static void save(IMObject... objects) {
        save(Arrays.asList(objects));
    }

    /**
     * Helper to save a collection of objects.
     *
     * @param objects the objects to save
     * @throws ArchetypeServiceException if the service cannot save the object
     * @throws ValidationException       if the object cannot be validated
     */
    public static void save(Collection<? extends IMObject> objects) {
        ArchetypeServiceHelper.getArchetypeService().save(objects);
    }

    /**
     * Creates a new customer with default contacts.
     *
     * @param firstName the customer's first name
     * @param lastName  the customer's surname
     * @param save      if {@code true} make the customer persistent
     * @return a new customer
     */
    public static Party createCustomer(String firstName, String lastName, boolean save) {
        return createCustomer(firstName, lastName, true, save);
    }

    /**
     * Creates a new customer.
     *
     * @param firstName       the customer's first name
     * @param lastName        the customer's surname
     * @param defaultContacts if {@code true}, add default contacts
     * @param save            if {@code true} make the customer persistent
     * @return a new customer
     */
    public static Party createCustomer(String firstName, String lastName, boolean defaultContacts, boolean save) {
        Party customer = create(CustomerArchetypes.PERSON, Party.class);
        if (defaultContacts) {
            PartyRules rules = new PartyRules(ArchetypeServiceHelper.getArchetypeService(),
                                              LookupServiceHelper.getLookupService());
            for (Contact contact : rules.getDefaultContacts()) {
                customer.addContact(contact);
            }
        }
        IMObjectBean bean = new IMObjectBean(customer);
        bean.setValue("firstName", firstName);
        bean.setValue("lastName", lastName);
        if (save) {
            bean.save();
        }
        return customer;
    }

    /**
     * Creates and saves a new customer.
     *
     * @return a new customer
     */
    public static Party createCustomer() {
        return createCustomer(true);
    }

    /**
     * Creates and saves a new customer, with the specified contacts.
     *
     * @param contacts the contacts
     * @return a new customer
     */
    public static Party createCustomer(Contact... contacts) {
        return createCustomer("MR", "J", randomName("Zoo-"), contacts);
    }

    /**
     * Creates and saves a new customer, with the specified contacts.
     *
     * @param titleCode the <em>lookup.personTitle</em> code. May be {@code null}
     * @param firstName the customer's first name
     * @param lastName  the customer's surname
     * @param contacts  the contacts
     * @return a new customer
     */
    public static Party createCustomer(String titleCode, String firstName, String lastName, Contact... contacts) {
        Party customer = createCustomer(firstName, lastName, false, false);
        if (titleCode != null) {
            IMObjectBean bean = new IMObjectBean(customer);
            bean.setValue("title", getLookup("lookup.personTitle", titleCode).getCode());
        }
        for (Contact contact : contacts) {
            customer.addContact(contact);
        }
        save(customer);
        return customer;
    }

    /**
     * Creates and saves a new customer.
     *
     * @param titleCode   the <em>lookup.personTitle</em> code. May be {@code null}
     * @param firstName   the first name
     * @param lastName    the last name
     * @param address     the address
     * @param suburbCode  the <em>lookup.suburb</em> code
     * @param stateCode   the <em>lookup.state</em> code
     * @param postCode    the post code
     * @param homePhone   the home phone. May be {@code null}
     * @param workPhone   the work phone. May be {@code null}
     * @param mobilePhone the mobile phone. May be {@code null}
     * @param email       the email address. May be {@code null}
     * @return a new customer
     */
    public static Party createCustomer(String titleCode, String firstName, String lastName,
                                       String address, String suburbCode, String stateCode, String postCode,
                                       String homePhone, String workPhone, String mobilePhone, String email) {
        Party customer = createCustomer(firstName, lastName, false, false);
        IMObjectBean bean = new IMObjectBean(customer);
        if (titleCode != null) {
            bean.setValue("title", getLookup("lookup.personTitle", titleCode).getCode());
        }
        customer.addContact(createLocationContact(address, suburbCode, stateCode, postCode));

        if (homePhone != null) {
            customer.addContact(createPhoneContact(null, homePhone, false, false, ContactArchetypes.HOME_PURPOSE));
        }
        if (workPhone != null) {
            customer.addContact(createPhoneContact(null, workPhone, false, false, ContactArchetypes.WORK_PURPOSE));
        }
        if (mobilePhone != null) {
            customer.addContact(createPhoneContact(null, mobilePhone, true));
        }
        if (email != null) {
            customer.addContact(createEmailContact(email));
        }
        save(customer);
        return customer;
    }

    /**
     * Creates a new customer.
     *
     * @param save if {@code true} make the customer persistent
     * @return a new customer
     */
    public static Party createCustomer(boolean save) {
        return createCustomer("J", randomName("Zoo-"), save);
    }

    /**
     * Creates a customer linked to a location.
     *
     * @param location the location
     * @return the customer
     */
    public static Party createCustomer(Party location) {
        Party customer = createCustomer();
        IMObjectBean bean = new IMObjectBean(customer);
        bean.setTarget("practice", location);
        bean.save();
        return customer;
    }

    /**
     * Creates a new <em>contact.location</em>
     * <p>
     * Any required lookups will be created and saved.
     *
     * @param address    the street address
     * @param suburbCode the <em>lookup.suburb</em> code
     * @param suburbName the suburb name. May be {@code null}
     * @param stateCode  the <em>lookup.state</em> code
     * @param stateName  the state name. May be {@code null}
     * @param postCode   the post code
     * @return a new location contact
     */
    public static Contact createLocationContact(String address, String suburbCode, String suburbName,
                                                String stateCode, String stateName, String postCode) {
        Lookup state = getLookup("lookup.state", stateCode, stateName, true);
        Lookup suburb = getLookup("lookup.suburb", suburbCode, suburbName, state, "lookupRelationship.stateSuburb");
        Contact contact = create(ContactArchetypes.LOCATION, Contact.class);
        IMObjectBean bean = new IMObjectBean(contact);
        bean.setValue("address", address);
        bean.setValue("suburb", suburb.getCode());
        bean.setValue("state", state.getCode());
        bean.setValue("postcode", postCode);
        return contact;
    }

    /**
     * Creates a new <em>contact.phoneNumber</em>
     *
     * @param areaCode the area code
     * @param number   the phone number
     * @return a new phone contact
     */
    public static Contact createPhoneContact(String areaCode, String number) {
        return createPhoneContact(areaCode, number, false);
    }

    /**
     * Creates a new <em>contact.phoneNumber</em>
     *
     * @param areaCode the area code
     * @param number   the phone number
     * @param sms      if {@code true}, allow SMS
     * @return a new phone contact
     */
    public static Contact createPhoneContact(String areaCode, String number, boolean sms) {
        return createPhoneContact(areaCode, number, sms, true, null);
    }

    /**
     * Creates a new <em>contact.phoneNumber</em>
     *
     * @param areaCode  the area code
     * @param number    the phone number
     * @param sms       if {@code true}, allow SMS
     * @param preferred if {@code true}, flags the contact as preferred
     * @param purpose   the contact purpose. May be {@code null}
     * @return a new phone contact
     */
    public static Contact createPhoneContact(String areaCode, String number, boolean sms, boolean preferred,
                                             String purpose) {
        Contact contact = create(ContactArchetypes.PHONE, Contact.class);
        IMObjectBean bean = new IMObjectBean(contact);
        bean.setValue("areaCode", areaCode);
        bean.setValue("telephoneNumber", number);
        bean.setValue("sms", sms);
        bean.setValue("preferred", preferred);
        if (purpose != null) {
            contact.addClassification(getLookup(ContactArchetypes.PURPOSE, purpose));
        }
        return contact;
    }

    /**
     * Creates a new <em>contact.email</em>
     *
     * @param address the email address
     * @return a new email contact
     */
    public static Contact createEmailContact(String address) {
        return createEmailContact(address, true, null);
    }

    /**
     * Creates a new <em>contact.email</em>
     *
     * @param address   the email address
     * @param preferred if {@code true}, flags the contact as preferred
     * @param purpose   the contact purpose. May be {@code null}
     * @return a new email contact
     */
    public static Contact createEmailContact(String address, boolean preferred, String purpose) {
        Contact contact = create(ContactArchetypes.EMAIL, Contact.class);
        IMObjectBean bean = new IMObjectBean(contact);
        bean.setValue("emailAddress", address);
        bean.setValue("preferred", preferred);
        if (purpose != null) {
            contact.addClassification(getLookup(ContactArchetypes.PURPOSE, purpose));
        }
        return contact;
    }

    /**
     * Creates and saves a new <em>contact.location</em>
     * <p>
     * Any required lookups will be created and saved.
     *
     * @param address    the street address
     * @param suburbCode the <em>lookup.suburb</em> code
     * @param stateCode  the <em>lookup.state</em> code
     * @param postCode   the post code
     * @return a new location contact
     */
    public static Contact createLocationContact(String address, String suburbCode, String stateCode, String postCode) {
        return createLocationContact(address, suburbCode, null, stateCode, null, postCode);
    }

    /**
     * Creates and saves a new patient, with species='CANINE'.
     *
     * @return a new patient
     */
    public static Party createPatient() {
        return createPatient(true);
    }

    /**
     * Creates a new patient, with species='CANINE'.
     *
     * @param save if {@code true} make the patient persistent
     * @return a new patient
     */
    public static Party createPatient(boolean save) {
        return createPatient(randomName("XPatient-"), save);
    }

    /**
     * Creates a new patient, with species='CANINE'.
     *
     * @param name the patient name
     * @param save if {@code true} make the patient persistent
     * @return a new patient
     */
    public static Party createPatient(String name, boolean save) {
        Party patient = create(PatientArchetypes.PATIENT, Party.class);
        IMObjectBean bean = new IMObjectBean(patient);
        bean.setValue("name", name);
        bean.setValue("species", "CANINE");
        bean.setValue("deceased", false);
        if (save) {
            bean.save();
        }
        return patient;
    }

    /**
     * Creates and saves a new patient, owned by the specified customer.
     *
     * @param owner the patient owner
     * @return a new patient
     */
    public static Party createPatient(Party owner) {
        return createPatient(owner, true);
    }

    /**
     * Creates a new patient, owned by the specified customer.
     *
     * @param owner the patient owner
     * @param save  if {@code true}, make the patient persistent
     * @return a new patient
     */
    public static Party createPatient(Party owner, boolean save) {
        return createPatient(randomName("XPatient-"), owner, save);
    }

    /**
     * Creates a new patient, owned by the specified customer.
     *
     * @param name  the patient name
     * @param owner the patient owner
     * @param save  if {@code true}, make the patient persistent
     * @return a new patient
     */
    public static Party createPatient(String name, Party owner, boolean save) {
        Party patient = createPatient(name, save);
        IArchetypeService service = ArchetypeServiceHelper.getArchetypeService();
        PatientRules rules = new PatientRules(null, null, service, null, null);
        rules.addPatientOwnerRelationship(owner, patient);
        if (save) {
            save(owner, patient);
        }
        return patient;
    }

    /**
     * Creates and saves a new user.
     *
     * @return a new user
     */
    public static User createUser() {
        return createUser(true);
    }

    /**
     * Creates a new user.
     *
     * @param save if {@code true} make the user persistent
     * @return a new user
     */
    public static User createUser(boolean save) {
        return createUser(randomName("zuser"), save);
    }

    /**
     * Creates a new user.
     *
     * @param username the login name
     * @param save     if {@code true} make the user persistent
     * @return a new user
     */
    public static User createUser(String username, boolean save) {
        return createUser(username, null, null, save);
    }

    /**
     * Creates a new user.
     *
     * @param firstName the first name
     * @param lastName  the last name
     * @return a new user
     */
    public static User createUser(String firstName, String lastName) {
        return createUser(randomName("user"), firstName, lastName, true);
    }

    /**
     * Creates a new user.
     *
     * @param username  the login name
     * @param firstName the first name. May be {@code null}
     * @param lastName  the last name. May be {@code null}
     * @param save      if {@code true} make the user persistent
     * @return a new user
     */
    public static User createUser(String username, String firstName, String lastName, boolean save) {
        User user = create(UserArchetypes.USER, User.class);
        IMObjectBean bean = new IMObjectBean(user);
        bean.setValue("name", username);
        bean.setValue("username", username);
        bean.setValue("firstName", firstName);
        bean.setValue("lastName", lastName);
        if (!StringUtils.isEmpty(firstName) && !StringUtils.isEmpty(lastName)) {
            bean.setValue("name", firstName + " " + lastName);
        }
        bean.setValue("password", username);
        if (save) {
            bean.save();
        }
        return user;
    }

    /**
     * Creates and saves a new clinician.
     *
     * @return a new clinician
     */
    public static User createClinician() {
        return createClinician(true);
    }

    /**
     * Creates a new clinician.
     *
     * @param save if {@code true} make the user persistent
     * @return a new user
     */
    public static User createClinician(boolean save) {
        String username = randomName("zuser");
        User user = createUser(username, false);
        user.addClassification(getLookup(UserArchetypes.USER_TYPE, UserArchetypes.CLINICIAN_USER_TYPE));
        if (save) {
            save(user);
        }
        return user;
    }

    /**
     * Creates a new administrator.
     *
     * @param save if {@code true} make the user persistent
     * @return a new user
     */
    public static User createAdministrator(boolean save) {
        String username = randomName("zuser");
        User user = createUser(username, false);
        user.addClassification(getLookup(UserArchetypes.USER_TYPE, UserArchetypes.ADMINISTRATOR_USER_TYPE));
        if (save) {
            save(user);
        }
        return user;
    }

    /**
     * Creates a new <em>product.medication</em> with no species classification.
     * The product name is prefixed with <em>XProduct-</em>.
     *
     * @return a new product
     */
    public static Product createProduct() {
        return createProduct(null);
    }

    /**
     * Creates a new <em>product.medicication</em> with an optional species
     * classification. The product name is prefixed with <em>XProduct-</em>.
     *
     * @param species the species classification. May be {@code null}
     * @return a new product
     */
    public static Product createProduct(String species) {
        return createProduct(ProductArchetypes.MEDICATION, species);
    }

    /**
     * Creates and saves a new product with an optional species classification.
     * The product name is prefixed with <em>XProduct-</em>.
     *
     * @param shortName the archetype short name
     * @param species   the species classification name. May be {@code null}
     * @return a new product
     */
    public static Product createProduct(String shortName, String species) {
        return createProduct(shortName, species, true);
    }

    /**
     * Creates a new product with an optional species classification.
     * The product name is prefixed with <em>XProduct-</em>.
     *
     * @param shortName the product short name
     * @param species   the species classification name. May be {@code null}
     * @param save      if {@code true}, save the product
     * @return a new product
     */
    public static Product createProduct(String shortName, String species, boolean save) {
        Product product = create(shortName, Product.class);
        IMObjectBean bean = new IMObjectBean(product);
        String name = randomName("XProduct-" + ((species != null) ? species : ""));
        bean.setValue("name", name);
        if (species != null) {
            Lookup classification
                    = getLookup("lookup.species", species);
            bean.addValue("species", classification);
        }
        if (save) {
            bean.save();
        }
        return product;
    }

    /**
     * Creates and saves new <em>party.supplierorganisation</em>.
     *
     * @return a new party
     */
    public static Party createSupplier() {
        return createSupplier(true);
    }

    /**
     * Creates a new <em>party.supplierorganisation</em>.
     *
     * @param save if {@code true} save the supplier
     * @return a new party
     */
    public static Party createSupplier(boolean save) {
        Party party = create(SupplierArchetypes.SUPPLIER_ORGANISATION, Party.class);
        IMObjectBean bean = new IMObjectBean(party);
        bean.setValue("name", "XSupplier");
        if (save) {
            bean.save();
        }
        return party;
    }

    /**
     * Creates a new <em>party.supplierVeterinarian</em>.
     *
     * @return a new party
     */
    public static Party createSupplierVet() {
        Party party = create(SupplierArchetypes.SUPPLIER_VET, Party.class);
        IMObjectBean bean = new IMObjectBean(party);
        bean.setValue("firstName", "J");
        bean.setValue("lastName", "XSupplierVet");
        bean.setValue("title", "MR");
        bean.save();
        return party;
    }

    /**
     * Creates a new <em>party.supplierVeterinaryPractice</em>.
     *
     * @return a new party
     */
    public static Party createSupplierVetPractice() {
        Party party = create(SupplierArchetypes.SUPPLIER_VET_PRACTICE, Party.class);
        party.setName("XVetPractice");
        save(party);
        return party;
    }

    /**
     * Returns the <em>party.organisationPractice</em> singleton,
     * creating one if it doesn't exist.
     * <p>
     * If it exists, any tax rates will be removed.
     * <p>
     * The practice currency is set to <em>AUD</em>.
     * <p>
     * Default contacts are added.
     *
     * @return the practice
     */
    public static Party getPractice() {
        Party party;
        ArchetypeQuery query = new ArchetypeQuery(PracticeArchetypes.PRACTICE, true, true);
        query.setMaxResults(1);
        QueryIterator<Party> iter = new IMObjectQueryIterator<>(query);
        if (iter.hasNext()) {
            party = iter.next();

            // remove any taxes
            IMObjectBean bean = new IMObjectBean(party);
            List<Lookup> taxes = bean.getValues("taxes", Lookup.class);
            if (!taxes.isEmpty()) {
                for (Lookup tax : taxes) {
                    bean.removeValue("taxes", tax);
                }
            }
        } else {
            party = create(PracticeArchetypes.PRACTICE, Party.class);
            party.setName("XPractice");
        }

        PartyRules rules = new PartyRules(ArchetypeServiceHelper.getArchetypeService(),
                                          LookupServiceHelper.getLookupService());
        for (Contact contact : rules.getDefaultContacts()) {
            party.addContact(contact);
        }

        Lookup currency = getCurrency("AUD");

        IMObjectBean bean = new IMObjectBean(party);
        bean.setValue("currency", currency.getCode());
        bean.setValue("useLocationProducts", false);
        bean.setValue("useLoggedInClinician", true);
        bean.save();
        return party;
    }

    /**
     * Returns the <em>party.organisationPractice</em> singleton creating one if it doesn't exist.
     *
     * @param rate the default tax rate
     * @return the practice
     */
    public static Party getPractice(BigDecimal rate) {
        Party practice = TestHelper.getPractice();
        Lookup tax = create("lookup.taxType", Lookup.class);
        IMObjectBean taxBean = new IMObjectBean(tax);
        taxBean.setValue("code", "XTAXTYPE" + Math.abs(new Random().nextInt()));
        taxBean.setValue("rate", rate);
        taxBean.save();
        practice.addClassification(tax);
        save(practice);
        return practice;
    }

    /**
     * Returns a currency with the specified currency code, creating it if it doesn't exist.
     *
     * @param code the currency code
     * @return the currency
     */
    public static Lookup getCurrency(String code) {
        Lookup currency = getLookup("lookup.currency", code, false);
        IMObjectBean ccyBean = new IMObjectBean(currency);
        ccyBean.setValue("minDenomination", new BigDecimal("0.05"));
        ccyBean.setValue("minPrice", null);
        ccyBean.save();
        return currency;
    }

    /**
     * Creates a new <em>party.organisationLocation</em>.
     *
     * @return a new location
     */
    public static Party createLocation() {
        return createLocation(false);
    }

    /**
     * Creates a new <em>party.organisationLocation</em>.
     *
     * @param stockControl if {@code true}, enable stock control for the location
     * @return a new location
     */
    public static Party createLocation(boolean stockControl) {
        return createLocation(null, null, stockControl);
    }

    /**
     * Creates a new <em>party.organisationLocation</em>.
     *
     * @param phone        the phone number. May be {@code null}
     * @param email        the email address. May be {@code null}
     * @param stockControl if {@code true}, enable stock control for the location
     * @return a new location
     */
    public static Party createLocation(String phone, String email, boolean stockControl) {
        Party location = create(PracticeArchetypes.LOCATION, Party.class);
        location.setName("XLocation");
        IMObjectBean bean = new IMObjectBean(location);
        bean.setValue("stockControl", stockControl);
        if (phone != null) {
            location.addContact(createPhoneContact(null, phone, false));
        }
        if (email != null) {
            location.addContact(createEmailContact(email));
        }
        save(location);
        return location;
    }

    /**
     * Creates a new till.
     *
     * @return the new till
     */
    public static Entity createTill() {
        Entity till = create(TillArchetypes.TILL, Entity.class);
        till.setName("TillRulesTestCase-Till" + till.hashCode());
        save(till);
        return till;
    }

    /**
     * Creates a new till linked to a location.
     *
     * @param location the practice location
     * @return the new till
     */
    public static Entity createTill(Party location) {
        Entity till = createTill();
        IMObjectBean bean = new IMObjectBean(location);
        bean.addTarget("tills", till, "locations");
        bean.save(till);
        return till;
    }


    /**
     * Returns a lookup, creating and saving it if it doesn't exist.
     *
     * @param shortName the lookup short name
     * @param code      the lookup code
     * @return the lookup
     */
    public static Lookup getLookup(String shortName, String code) {
        return getLookup(shortName, code, true);
    }

    /**
     * Returns a lookup, creating it if it doesn't exist.
     *
     * @param shortName the lookup short name
     * @param code      the lookup code
     * @param save      if {@code true}, save the lookup
     * @return the lookup
     */
    public static Lookup getLookup(String shortName, String code, boolean save) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, false);
        query.add(new NodeConstraint("code", code));
        query.setMaxResults(1);
        QueryIterator<Lookup> iter = new IMObjectQueryIterator<>(query);
        if (iter.hasNext()) {
            return iter.next();
        }
        Lookup lookup = create(shortName, Lookup.class);
        lookup.setCode(code);
        if (save) {
            save(lookup);
        }
        return lookup;
    }

    /**
     * Returns a lookup, creating it if it doesn't exist.
     * <p>
     * If the lookup exists, but the name is different, the name will be updated.
     *
     * @param shortName the lookup short name
     * @param code      the lookup code
     * @param name      the lookup name
     * @param save      if {@code true}, save the lookup
     * @return the lookup
     */
    public static Lookup getLookup(String shortName, String code, String name, boolean save) {
        Lookup lookup;
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, false);
        query.add(new NodeConstraint("code", code));
        query.setMaxResults(1);
        QueryIterator<Lookup> iter = new IMObjectQueryIterator<>(query);
        if (iter.hasNext()) {
            lookup = iter.next();
            if (!StringUtils.equals(name, lookup.getName()) || !lookup.isActive()) {
                lookup.setName(name);
                lookup.setActive(true);
            } else {
                save = false;
            }
        } else {
            lookup = create(shortName, Lookup.class);
            lookup.setCode(code);
            lookup.setName(name);
        }
        if (save) {
            save(lookup);
        }
        return lookup;
    }

    /**
     * Returns a lookup that is the target in a lookup relationship, creating and saving it if it doesn't exist.
     *
     * @param shortName             the target lookup short name
     * @param code                  the lookup code
     * @param source                the source lookup
     * @param relationshipShortName the lookup relationship short name
     * @return the lookup
     */
    public static Lookup getLookup(String shortName, String code, Lookup source,
                                   String relationshipShortName) {
        return getLookup(shortName, code, code, source, relationshipShortName);
    }

    /**
     * Returns a lookup that is the target in a lookup relationship, creating
     * and saving it if it doesn't exist.
     *
     * @param shortName             the target lookup short name
     * @param code                  the lookup code
     * @param name                  the lookup name
     * @param source                the source lookup
     * @param relationshipShortName the lookup relationship short name
     * @return the lookup
     */
    public static Lookup getLookup(String shortName, String code, String name, Lookup source,
                                   String relationshipShortName) {
        Lookup target = getLookup(shortName, code, name, true);
        for (Relationship relationship : source.getLookupRelationships()) {
            if (relationship.getTarget().equals(target.getObjectReference())) {
                return target;
            }
        }
        LookupRelationship relationship = create(relationshipShortName, LookupRelationship.class);
        relationship.setSource(source.getObjectReference());
        relationship.setTarget(target.getObjectReference());
        source.addLookupRelationship(relationship);
        target.addLookupRelationship(relationship);
        save(source, target);
        return target;
    }

    /**
     * Helper to return the name of the lookup for the specified object and
     * node.
     *
     * @param object the object
     * @param node   the lookup node
     * @return the corresponding lookup's name. May be {@code null}
     */
    public static String getLookupName(IMObject object, String node) {
        IMObjectBean bean = new IMObjectBean(object);
        return LookupHelper.getName(ArchetypeServiceHelper.getArchetypeService(),
                                    LookupServiceHelper.getLookupService(), bean.getDescriptor(node),
                                    object);
    }

    /**
     * Helper to create and save a new tax type classification.
     *
     * @return a new tax classification
     */
    public static Lookup createTaxType(BigDecimal rate) {
        Lookup tax = create("lookup.taxType", Lookup.class);
        IMObjectBean bean = new IMObjectBean(tax);
        bean.setValue("code", "XTAXTYPE" + System.nanoTime());
        bean.setValue("rate", rate);
        save(tax);
        return tax;
    }

    /**
     * Creates a new act identity.
     *
     * @param archetype the identity archetype
     * @param id        the identity
     * @return a new act identity
     */
    public static ActIdentity createActIdentity(String archetype, String id) {
        ActIdentity identity = create(archetype, ActIdentity.class);
        identity.setIdentity(id);
        return identity;
    }

    /**
     * Creates a new entity identity.
     *
     * @param archetype the identity archetype
     * @param id        the identity
     * @return a new entity identity
     */
    public static EntityIdentity createEntityIdentity(String archetype, String id) {
        EntityIdentity identity = create(archetype, EntityIdentity.class);
        identity.setIdentity(id);
        return identity;
    }

    /**
     * Helper to create a date-time given a string of the form
     * <em>yyyy-mm-dd hh:mm:ss</em> or <em>yyyy-mm-dd hh:mm</em>
     *
     * @param value the value. May be {@code null}
     * @return the corresponding date-time or {@code null} if {@code value} is null
     */
    public static Date getDatetime(String value) {
        return getDatetime(value, false);
    }

    /**
     * Helper to create a date-time given a string of the form
     * <em>yyyy-mm-dd hh:mm:ss</em> or <em>yyyy-mm-dd hh:mm</em>
     * <p/>
     * The {@code useTimestamp} flag can be used to select the appropriate return type, if dates need to be compared.
     *
     * @param value        the value. May be {@code null}
     * @param useTimestamp if {@code true}, return a {@code java.sql.Timestamp} else return a {@code java.util.Data}.
     * @return the corresponding date-time or {@code null} if {@code value} is null
     */
    public static Date getDatetime(String value, boolean useTimestamp) {
        if (value != null) {
            if (StringUtils.countMatches(value, ":") != 2) {
                value = value + ":00";
            }
            Timestamp timestamp = Timestamp.valueOf(value);
            return (useTimestamp) ? timestamp : new Date(timestamp.getTime()); // use Date, for easy comparison
        }
        return null;
    }

    /**
     * Helper to create a date given a string of the form <em>yyyy-mm-dd</em>.
     *
     * @param value the value. May be {@code null}
     * @return the corresponding date, or {@code null} if {@code value} is null
     */
    public static Date getDate(String value) {
        return value != null ? getDatetime(value + " 0:0:0") : null;
    }

    /**
     * Parses a date or date time.
     *
     * @param value the value. May be {@code null}
     * @return the corresponding date. May be {@code null}
     */
    public static Date parseDate(String value) {
        if (value != null) {
            return value.contains(":") ? TestHelper.getDatetime(value) : TestHelper.getDate(value);
        }
        return null;
    }

    /**
     * Verifies two {@code BigDecimal} instances are equal.
     *
     * @param expected the expected value. May be {@code null}
     * @param actual   the actual value. May be {@code null}
     */
    public static void checkEquals(BigDecimal expected, BigDecimal actual) {
        if (expected == null) {
            assertNull(actual);
        } else if (actual == null || expected.compareTo(actual) != 0) {
            fail("Expected " + expected + ", but got " + actual);
        }
    }

    /**
     * Creates a name starting with the specified prefix, with a random numerical suffix.
     *
     * @param prefix the prefix
     * @return a random name
     */
    public static String randomName(String prefix) {
        return prefix + Math.abs(random.nextInt());
    }

    /**
     * Verifies all expected objects are included in a list. The list may contain additional objects.
     *
     * @param list     the list
     * @param included the expected objects to be included
     * @return the matching objects, in the order they appear
     */
    @SafeVarargs
    public static <T, R> List<R> assertIncluded(Iterable<R> list, T... included) {
        return assertIncluded(IterableUtils.toList(list), included);
    }

    /**
     * Verifies all expected objects are included in a list. The list may contain additional objects.
     *
     * @param list     the list
     * @param included the expected objects to be included
     * @return the matching objects, in the order they appear
     */
    @SafeVarargs
    @SuppressWarnings({"SuspiciousMethodCalls"})
    public static <T, R> List<R> assertIncluded(List<R> list, T... included) {
        List<R> result = new ArrayList<>(list);
        List<T> expected = Arrays.asList(included);
        result.retainAll(expected);
        assertEquals(expected.size(), result.size());
        return result;
    }

    /**
     * Verifies all the specified objects are not present in a list.
     *
     * @param list     the list
     * @param excluded the objects that should not be present
     */
    @SafeVarargs
    public static <T, R> void assertExcluded(Iterable<T> list, R... excluded) {
        assertExcluded(IterableUtils.toList(list), excluded);
    }

    /**
     * Verifies all the specified objects are not present in a list.
     *
     * @param list     the list
     * @param excluded the objects that should not be present
     */
    @SafeVarargs
    @SuppressWarnings({"unchecked", "SuspiciousMethodCalls"})
    public static <T, R> void assertExcluded(List<T> list, R... excluded) {
        for (R exclude : excluded) {
            assertFalse(list.contains(exclude));
        }
    }
    /**
     * Sets the identifier for an object, for testing purposes.
     * <p/>
     * This can be used where objects won't be stored.
     *
     * @param object the object
     * @param id     the object identifier
     */
    public static void setId(IMObject object, long id) {
        ((org.openvpms.component.business.domain.im.common.IMObject) object).setId(id);
    }
}
