/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.supplier;

import org.junit.Test;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.DELIVERY;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.DELIVERY_ITEM;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.ORDER;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.ORDER_ITEM;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.RETURN;
import static org.openvpms.archetype.rules.supplier.SupplierArchetypes.RETURN_ITEM;


/**
 * Tests the <em>act.supplierAccountInvoice</em>, <em>act.supplierAccountCredit<em>
 * <em>act.supplierOrder</em>, <em>act.supplierDelivery</em> and <em>act.supplierReturn</em> archetypes.
 *
 * @author Tim Anderson
 */
public class SupplierActTestCase extends AbstractSupplierTest {

    /**
     * Verifies that when a supplier invoice is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteInvoice() {
        checkDelete(SupplierArchetypes.INVOICE, SupplierArchetypes.INVOICE_ITEM);
    }

    /**
     * Verifies that when a supplier credit is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteCredit() {
        checkDelete(SupplierArchetypes.CREDIT, SupplierArchetypes.CREDIT_ITEM);
    }

    /**
     * Verifies that when a supplier payment is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeletePayment() {
        checkDelete(SupplierArchetypes.PAYMENT, SupplierArchetypes.PAYMENT_CASH);
    }

    /**
     * Verifies that when a supplier refund is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteRefund() {
        checkDelete(SupplierArchetypes.REFUND, SupplierArchetypes.REFUND_CASH);
    }

    /**
     * Verifies that when a supplier order is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteOrder() {
        checkDelete(ORDER, ORDER_ITEM);
    }

    /**
     * Verifies that when a supplier delivery is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteDelivery() {
        checkDelete(DELIVERY, DELIVERY_ITEM);
    }

    /**
     * Verifies that when a supplier delivery is deleted, the linked order item is not deleted.
     */
    @Test
    public void testDeleteDeliveryLinkedToOrder() {
        Party supplier = getSupplier();
        Product product = getProduct();
        List<Act> order = createActs(supplier, product, ORDER, ORDER_ITEM);
        List<Act> delivery = createActs(supplier, product, DELIVERY, DELIVERY_ITEM);

        Act deliveryItem = delivery.get(1);
        IMObjectBean deliveryItemBean = getBean(deliveryItem);
        Act orderItem = order.get(1);
        deliveryItemBean.addTarget("order", orderItem, "deliveries");
        save(deliveryItem, orderItem);

        checkDelete(delivery);
        assertNotNull(get(order.get(0)));
        assertNotNull(get(orderItem));
    }

    /**
     * Verifies that when a supplier return is deleted, any child act is deleted with it.
     */
    @Test
    public void testDeleteReturn() {
        checkDelete(RETURN, RETURN_ITEM);
    }

    /**
     * Verifies that when a supplier return is deleted, the linked order item is not deleted.
     */
    @Test
    public void testDeleteReturnLinkedToOrder() {
        Party supplier = getSupplier();
        Product product = getProduct();
        List<Act> order = createActs(supplier, product, ORDER, ORDER_ITEM);
        List<Act> supplierReturn = createActs(supplier, product, RETURN, RETURN_ITEM);

        Act returnItem = supplierReturn.get(1);
        IMObjectBean returnItemBean = getBean(returnItem);
        Act orderItem = order.get(1);
        returnItemBean.addTarget("order", orderItem, "return");
        save(returnItem, orderItem);

        checkDelete(supplierReturn);
        assertNotNull(get(order.get(0)));
        assertNotNull(get(orderItem));
    }

    /**
     * Verifies that when a supplier act is deleted, any child act is deleted with it.
     *
     * @param shortName     the parent act archetype short name
     * @param itemShortName the child act archetype short name
     */
    private void checkDelete(String shortName, String itemShortName) {
        List<Act> acts = createActs(getSupplier(), getProduct(), shortName, itemShortName);

        checkDelete(acts);
    }

    /**
     * Verifies that when a supplier act is deleted, any child act is deleted with it.
     *
     * @param acts the parent and child acts
     */
    private void checkDelete(List<Act> acts) {
        Act parent = acts.get(0);
        Act child = acts.get(1);

        assertNotNull(get(parent));
        assertNotNull(get(child));

        remove(parent);
        assertNull(get(parent));
        assertNull(get(child));
    }

    /**
     * Creates a parent supplier act linked to a child act.
     *
     * @param supplier      the supplier
     * @param product       the product
     * @param shortName     the parent act archetype short name
     * @param itemShortName the child act archetype short name
     * @return the two acts
     */
    private List<Act> createActs(Party supplier, Product product, String shortName, String itemShortName) {
        IMObjectBean bean = createAct(shortName, supplier);
        Act act = (Act) bean.getObject();
        FinancialAct item = create(itemShortName, FinancialAct.class);
        IMObjectBean itemBean = getBean(item);
        if (itemBean.hasNode("product")) {
            itemBean.setTarget("product", product);
        }
        item.setQuantity(BigDecimal.ONE);
        getArchetypeService().deriveValues(item);
        ActRelationship relationship = (ActRelationship) bean.addTarget("items", item);
        item.addActRelationship(relationship);
        List<Act> acts = new ArrayList<>();
        acts.add(act);
        acts.add(item);
        save(acts);
        return acts;
    }

    /**
     * Creates a new supplier act.
     *
     * @param shortName the act short name
     * @param supplier  the supplier
     * @return a new act
     */
    protected IMObjectBean createAct(String shortName, Party supplier) {
        return createAct(shortName, supplier, getStockLocation());
    }

    /**
     * Creates a new supplier act.
     *
     * @param shortName the act short name
     * @param supplier  the supplier
     * @return a new act
     */
    protected IMObjectBean createAct(String shortName, Party supplier, Party stockLocation) {
        Act act = create(shortName, Act.class);
        IMObjectBean bean = getBean(act);
        bean.setTarget("supplier", supplier);
        if (bean.hasNode("stockLocation")) {
            bean.setTarget("stockLocation", stockLocation);
        }
        return bean;
    }
}