/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.patient;

import org.junit.Before;
import org.junit.Test;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.customer.TestCustomerFactory;
import org.openvpms.archetype.test.builder.customer.account.TestCustomerAccountFactory;
import org.openvpms.archetype.test.builder.doc.TestDocumentFactory;
import org.openvpms.archetype.test.builder.patient.TestPatientFactory;
import org.openvpms.archetype.test.builder.patient.TestVisitBuilder;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.archetype.test.builder.product.TestProductFactory;
import org.openvpms.archetype.test.builder.scheduling.TestSchedulingFactory;
import org.openvpms.archetype.test.builder.user.TestUserFactory;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.DocumentAct;
import org.openvpms.component.model.archetype.NodeDescriptor;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.user.User;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.openvpms.archetype.rules.act.ActStatus.COMPLETED;
import static org.openvpms.archetype.rules.act.ActStatus.IN_PROGRESS;
import static org.openvpms.archetype.test.TestHelper.getDate;
import static org.openvpms.archetype.test.TestHelper.getDatetime;

/**
 * Tests the {@link MedicalRecordRules} class.
 *
 * @author Tim Anderson
 */
public class MedicalRecordRulesTestCase extends ArchetypeServiceTest {

    /**
     * The customer account factory.
     */
    @Autowired
    private TestCustomerAccountFactory accountFactory;

    /**
     * The customer factory.
     */
    @Autowired
    private TestCustomerFactory customerFactory;

    /**
     * The document factory.
     */
    @Autowired
    private TestDocumentFactory documentFactory;

    /**
     * The patient factory.
     */
    @Autowired
    private TestPatientFactory patientFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * The product factory.
     */
    @Autowired
    private TestProductFactory productFactory;

    /**
     * The scheduling factory.
     */
    @Autowired
    private TestSchedulingFactory schedulingFactory;

    /**
     * The user factory.
     */
    @Autowired
    private TestUserFactory userFactory;

    /**
     * The patient.
     */
    private Party patient;

    /**
     * The clinician.
     */
    private User clinician;

    /**
     * The location.
     */
    private Party location;

    /**
     * The rules.
     */
    private MedicalRecordRules rules;

    /**
     * Sets up the test case.
     */
    @Before
    public void setUp() {
        clinician = userFactory.createClinician();
        patient = patientFactory.createPatient();
        location = practiceFactory.createLocation();
        rules = new MedicalRecordRules(getArchetypeService());
    }

    /**
     * Verifies that deletion of an <em>act.patientClinicalProblem</em>
     * doesn't affect its children.
     */
    @Test
    public void testDeleteClinicalProblem() {
        Act event = createEvent();
        Act problem = createProblem();
        Act note = createNote();
        patientFactory.updateVisit(event).addItems(problem, note).build();
        patientFactory.updateProblem(problem).addItem(note).build();

        // make sure each of the objects can be retrieved
        assertNotNull(get(event.getObjectReference()));
        assertNotNull(get(problem.getObjectReference()));

        remove(problem);  // remove shouldn't cascade to delete note

        // make sure the all but the problem can be retrieved
        assertNotNull(get(event));
        assertNull(get(problem));
        assertNotNull(get(note));
    }

    /**
     * Tests the {@link MedicalRecordRules#getEvent(Party)} method.
     */
    @Test
    public void testGetEvent() {
        Act event1 = createEvent(getDate("2007-01-01"), null, IN_PROGRESS);
        checkGetEvent(event1);

        Act event2 = createEvent(getDate("2007-01-02"), null, COMPLETED);
        checkGetEvent(event2);

        Act event3 = createEvent(getDate("2008-01-01"), null, IN_PROGRESS);
        checkGetEvent(event3);

        // ensure that where there are 2 events with the same timestamp, the one with the higher id is returned
        Act event4 = createEvent(getDate("2008-01-01"), null, IN_PROGRESS);
        checkGetEvent(event4);
    }

    /**
     * Tests the {@link MedicalRecordRules#getEvent} method.
     */
    @Test
    public void testGetEventByDate() {
        Date jan1 = getDate("2007-01-01");
        Date jan2 = getDate("2007-01-02");
        Date jan3 = getDatetime("2007-01-03 10:43:55");

        checkGetEvent(jan2, null);

        Act event1 = createEvent(jan2);
        checkGetEvent(jan2, event1);
        checkGetEvent(jan1, null);
        checkGetEvent(jan3, event1);

        event1.setActivityEndTime(jan2);
        save(event1);
        checkGetEvent(jan1, null);
        checkGetEvent(jan3, null);

        Act event2 = createEvent(jan3);
        checkGetEvent(jan3, event2);
        checkGetEvent(getDate("2007-01-03"), event2);
        // note that the time component is zero, but still picks up event2,
        // despite the event being created after 00:00:00. This is required
        // as the time component of startTime is not supplied consistently -
        // In some cases, it is present, in others it is 00:00:00.

        checkGetEvent(jan2, event1);

        // make sure that when an event has a duplicate timestamp, the earliest (by id) is returned
        Act event2dup = createEvent(jan3);
        save(event2dup);
        checkGetEvent(jan3, event2);
    }

    /**
     * Tests the {@link MedicalRecordRules#createNote} method.
     */
    @Test
    public void testCreateNote() {
        Date startTime = getDate("2012-07-17");
        String text = "Test note";
        Act note = rules.createNote(startTime, patient, text, clinician);

        IMObjectBean bean = getBean(note);
        assertEquals(startTime, note.getActivityStartTime());
        assertEquals(text, bean.getString("note"));
        assertEquals(patient, bean.getTarget("patient"));
        assertEquals(clinician, bean.getTarget("clinician"));
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where no event
     * exists for the patient. A new one will be created with IN_PROGRESS status,
     * and the specified startTime.
     */
    @Test
    public void testAddToEventForNonExistentEvent() {
        Date date = getDate("2007-04-05");
        Act medication = createMedication(patient);

        rules.addToEvent(medication, date, practiceFactory.createLocation());
        Act event = rules.getEvent(patient);
        assertEquals(IN_PROGRESS, event.getStatus());
        assertEquals(date, event.getActivityStartTime());
        checkContains(event, medication);
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where there is
     * an existing IN_PROGRESS event that has a startTime < 7 days prior to
     * that specified. The medication should be added to it.
     */
    @Test
    public void testAddToEventForExistingInProgressEvent() {
        Date date = getDate("2007-04-05");
        Act medication = patientFactory.newMedication()
                .patient(patient)
                .product(productFactory.createMedication())
                .build();

        Act expected = createEvent(date);
        save(expected);

        rules.addToEvent(medication, date, location);
        Act event = rules.getEvent(patient);
        checkContains(event, medication);
        assertEquals(expected, event);
        assertEquals(IN_PROGRESS, event.getStatus());
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where
     * there is an IN_PROGRESS event that has a startTime > 7 days prior to
     * the specified startTime. A new IN_PROGRESS event should be created.
     */
    @Test
    public void testAddToEventForExistingOldInProgressEvent() {
        Date date = getDate("2007-04-05");
        Act medication = createMedication(patient);

        Date old = DateRules.getDate(date, -8, DateUnits.DAYS);
        Act oldEvent = createEvent(old);
        save(oldEvent);

        rules.addToEvent(medication, date, location);
        Act event = rules.getEvent(patient);
        checkContains(event, medication);
        assertNotEquals(oldEvent, event);
        assertEquals(date, event.getActivityStartTime());
        assertEquals(IN_PROGRESS, event.getStatus());
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where
     * there is a COMPLETED event that has a startTime > 7 days prior to
     * the specified startTime. A new COMPLETED event should be created.
     */
    public void testAddToEventForExistingOldCompletedEvent() {
        Date date = getDate("2007-04-05");
        Act medication = createMedication(patient);

        Date old = DateRules.getDate(date, -8, DateUnits.DAYS);
        Act oldEvent = createEvent(old);
        oldEvent.setStatus(COMPLETED);
        save(oldEvent);

        rules.addToEvent(medication, date, location);
        Act event = rules.getEvent(patient);
        checkContains(event, medication);
        assertNotEquals(oldEvent, event);
        assertEquals(date, event.getActivityStartTime());
        assertEquals(COMPLETED, event.getStatus());
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where there is a COMPLETED event that has a startTime and
     * endTime that overlaps the specified start time. The medication should be added to it.
     */
    @Test
    public void testAddToEventForExistingCompletedEvent() {
        Date date = getDate("2007-04-05");
        Act medication = createMedication(patient);

        Act completed = createEvent(getDate("2007-04-03"), getDate("2007-04-06"), COMPLETED);

        rules.addToEvent(medication, date, location);
        Act event = rules.getEvent(patient);
        checkContains(event, medication);
        assertEquals(completed, event);
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvent} method where there is a {@code COMPLETED} event that has a
     * startTime and endTime that DOESN'T overlap the specified start time. The medication should be added
     * to a new {@code IN_PROGRESS} event whose startTime equals that specified.
     */
    @Test
    public void testAddToEventForExistingNonOverlappingCompletedEvent() {
        Date date = getDate("2007-04-05");
        Act medication = patientFactory.createMedication(patient, productFactory.createMedication());

        Act completed = createEvent(getDate("2007-04-03"));
        completed.setActivityEndTime(getDate("2007-04-04"));
        completed.setStatus(COMPLETED);
        save(completed);

        rules.addToEvent(medication, date, location);
        Act event = rules.getEvent(patient);
        checkContains(event, medication);
        assertNotEquals(completed, event);
        assertEquals(date, event.getActivityStartTime());
        assertEquals(IN_PROGRESS, event.getStatus());
    }

    /**
     * Verifies that a new event is created if the prior IN_PROGRESS event is for a different location.
     */
    @Test
    public void testGetEventForAdditionForInProgressEventAtDifferentLocationCreatesNewEvent() {
        Date start1 = DateRules.getYesterday();
        Date end1 = DateRules.getDate(start1, 1, DateUnits.HOURS);
        Act event1 = createEvent(start1, end1, IN_PROGRESS);
        Party location2 = practiceFactory.createLocation();
        Date start2 = DateRules.getToday();
        Act event2 = rules.getEventForAddition(patient, start2, null, location2);
        assertNotEquals(event1, event2);
        checkEvent(event2, start2, null, IN_PROGRESS, location2);
    }

    /**
     * Verifies that a new event is created if the prior COMPLETED event is for a different location.
     */
    @Test
    public void testGetEventForAdditionForCompletedEventAtDifferentLocationCreatesNewEvent() {
        Date start1 = DateRules.getYesterday();
        Date end1 = DateRules.getDate(start1, 1, DateUnits.HOURS);
        Act event1 = createEvent(start1, end1, COMPLETED);
        Party location2 = practiceFactory.createLocation();
        Date start2 = DateRules.getToday();
        Act event2 = rules.getEventForAddition(patient, start2, null, location2);
        assertNotEquals(event1, event2);
        checkEvent(event2, start2, null, IN_PROGRESS, location2);
    }

    /**
     * Verifies that the existing event is returned if it has no location, but a location was supplied.
     * <p/>
     * This is to support legacy events/plugins where no location is present on the event, but the same event
     * should be used.
     */
    @Test
    public void testGetEventForAdditionWhereExistingVisitHasNoLocation() {
        Date start1 = DateRules.getYesterday();
        Act event1 = newEvent(start1, null)
                .location(null)
                .build();
        assertEquals(IN_PROGRESS, event1.getStatus());

        Party location2 = practiceFactory.createLocation();
        Date start2 = DateRules.getToday();
        Act event2 = rules.getEventForAddition(patient, start2, null, location2);
        assertEquals(event1, event2);
        checkEvent(event2, start1, null, IN_PROGRESS, null); // NOTE: location not added

        // verify that if no location is specified, event1 is returned
        Act event3 = rules.getEventForAddition(patient, start2, null, null);
        assertEquals(event1, event3);
    }

    /**
     * Verifies that when no visit exists for a future date, the current time is used to create a new event.
     */
    @Test
    public void testGetEventForAdditionForFutureDate() {
        Date tomorrow = DateRules.getTomorrow();
        Act event1 = rules.getEventForAddition(patient, tomorrow, clinician, location);
        assertTrue(event1.isNew());
        Date now = new Date();
        assertTrue(event1.getActivityStartTime().compareTo(now) <= 0);
        save(event1);

        Act event2 = rules.getEventForAddition(patient, tomorrow, clinician, location);
        assertEquals(event1, event2);
    }

    /**
     * Tests the {@link org.openvpms.archetype.rules.patient.MedicalRecordRules#linkMedicalRecords} method.
     */
    @Test
    public void testLinkMedicalRecords() {
        Act event = createEvent();
        Act problem = createProblem();
        Act note = createNote();
        Act addendum = createAddendum();
        rules.linkMedicalRecords(event, problem, note, addendum);

        event = get(event);
        problem = get(problem);
        note = get(note);
        addendum = get(addendum);

        IMObjectBean eventBean = getBean(event);
        assertTrue(eventBean.hasTarget("items", problem));
        assertTrue(eventBean.hasTarget("items", note));
        assertTrue(eventBean.hasTarget("items", addendum));

        IMObjectBean problemBean = getBean(problem);
        assertTrue(problemBean.hasTarget("items", note));
        assertTrue(problemBean.hasTarget("items", addendum));

        // verify that it can be called again with no ill effect
        rules.linkMedicalRecords(event, problem, note, addendum);
    }

    /**
     * Tests the {@link MedicalRecordRules#linkMedicalRecords(Act, Act)} method
     * passing an <em>act.patientClinicalNote</em>.
     */
    @Test
    public void testLinkMedicalRecordsWithItem() {
        Act event = createEvent();
        Act note = createNote();

        rules.linkMedicalRecords(event, note);

        event = get(event);
        note = get(note);

        IMObjectBean eventBean = getBean(event);
        assertTrue(eventBean.hasTarget("items", note));
        assertEquals(1, event.getActRelationships().size());

        // verify that it can be called again with no ill effect
        rules.linkMedicalRecords(event, note);
        assertEquals(1, event.getActRelationships().size());
    }

    /**
     * Tests the {@link MedicalRecordRules#linkMedicalRecords(Act, Act)} method passing
     * an <em>act.customerAccountInvoiceItem</em>.
     */
    @Test
    public void testLinkMedicalRecordsWithInvoiceItem() {
        Act event = createEvent();
        Act invoiceItem = accountFactory.newInvoiceItem()
                .patient(patient)
                .product(productFactory.createMedication())
                .quantity(1)
                .build();
        rules.linkMedicalRecords(event, invoiceItem);

        event = get(event);
        invoiceItem = get(invoiceItem);

        IMObjectBean eventBean = getBean(event);
        assertTrue(eventBean.hasTarget("chargeItems", invoiceItem));
        assertEquals(1, event.getActRelationships().size());

        // verify that it can be called again with no ill effect
        rules.linkMedicalRecords(event, invoiceItem);
        assertEquals(1, event.getActRelationships().size());
    }

    /**
     * Tests the {@link MedicalRecordRules#linkMedicalRecords(Act, Act)} method,
     * passing an <em>act.patientClinicalProblem</em>.
     */
    @Test
    public void testLinkMedicalRecordsWithProblem() {
        Act event = createEvent();
        Act problem = createProblem();
        Act note = createNote();

        IMObjectBean problemBean = getBean(problem);
        problemBean.addTarget("items", note, "problem");
        save(problem, note);

        rules.linkMedicalRecords(event, problem);

        event = get(event);
        problem = get(problem);
        note = get(note);

        IMObjectBean eventBean = getBean(event);
        assertTrue(eventBean.hasTarget("items", note));
        assertTrue(eventBean.hasTarget("items", problem));
        assertEquals(2, event.getActRelationships().size());
        assertEquals(2, problem.getActRelationships().size());

        // verify that it can be called again with no ill effect
        rules.linkMedicalRecords(event, problem);
        assertEquals(2, event.getActRelationships().size());
        assertEquals(2, problem.getActRelationships().size());
    }

    /**
     * Verifies the {@link MedicalRecordRules#linkMedicalRecords(Act, Act, Act, Act)} method,
     * links all of a problem's items to the parent event if they aren't already present.
     */
    @Test
    public void testLinkMedicalRecordsForMissingLinks() {
        Act event = createEvent();
        Act problem = createProblem();
        Act note1 = createNote();
        Act note2 = createNote();
        Act medication = patientFactory.createMedication(patient, productFactory.createMedication());
        IMObjectBean problemBean = getBean(problem);
        problemBean.addTarget("items", note1, "problem");
        problemBean.addTarget("items", medication, "problem");
        save(problem, note1, medication);

        // now link the records to the event
        rules.linkMedicalRecords(event, problem, note2, null);

        event = get(event);
        problem = get(problem);
        note1 = get(note1);
        note2 = get(note2);
        medication = get(medication);

        IMObjectBean eventBean = getBean(event);
        assertTrue(eventBean.hasTarget("items", problem));
        assertTrue(eventBean.hasTarget("items", note1));
        assertTrue(eventBean.hasTarget("items", note2));
        assertTrue(eventBean.hasTarget("items", medication));

        problemBean = getBean(problem);
        assertTrue(problemBean.hasTarget("items", note1));
        assertTrue(problemBean.hasTarget("items", note2));
        assertTrue(problemBean.hasTarget("items", medication));
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvents} method.
     */
    @Test
    public void testAddToEvents() {
        Date date = getDate("2007-04-05");
        Party patient2 = patientFactory.createPatient();
        Act med1 = patientFactory.createMedication(patient, productFactory.createMedication());
        Act med2 = patientFactory.createMedication(patient, productFactory.createMedication());
        Act med3 = patientFactory.createMedication(patient2, productFactory.createMedication());
        Act med4 = patientFactory.createMedication(patient2, productFactory.createMedication());

        List<Act> acts = Arrays.asList(med1, med2, med3, med4);

        Act event1 = createEvent(date);
        save(event1);
        rules.addToEvents(acts, date, location);

        event1 = rules.getEvent(patient, date, location);
        checkContains(event1, med1, med2);

        Act event2 = rules.getEvent(patient2, date, location);
        assertNotNull(event2);
        checkContains(event2, med3, med4);
    }

    /**
     * Tests the {@link MedicalRecordRules#addToEvents} method where an event relationship already exists, but
     * to a different patient.
     */
    @Test
    public void testAddToEventsForDifferentPatient() {
        Date date = getDate("2014-03-22");
        Party patient2 = patientFactory.createPatient();
        Party patient3 = patientFactory.createPatient();
        Act med1 = patientFactory.createMedication(patient, productFactory.createMedication());
        Act med2 = patientFactory.createMedication(patient, productFactory.createMedication());
        Act med3 = patientFactory.createMedication(patient2, productFactory.createMedication());
        Act med4 = patientFactory.createMedication(patient2, productFactory.createMedication());

        List<Act> acts = Arrays.asList(med1, med2, med3, med4);

        Act event1 = createEvent(date);
        save(event1);
        rules.addToEvents(acts, date, location);

        event1 = rules.getEvent(patient, date, location);
        checkContains(event1, med1, med2);

        Act event2 = rules.getEvent(patient2, date, location);
        assertNotNull(event2);
        checkContains(event2, med3, med4);

        // now change the patient for med2 and med4 to patient3
        setPatient(med2, patient3);
        setPatient(med4, patient3);
        rules.addToEvents(acts, date, location);

        event1 = rules.getEvent(patient, date, location);
        assertNotNull(event1);
        checkContains(event1, med1);

        event2 = rules.getEvent(patient2, date, location);
        assertNotNull(event2);
        checkContains(event2, med3);

        Act event3 = rules.getEvent(patient3, date, location);
        assertNotNull(event3);
        checkContains(event3, med2, med4);
    }

    /**
     * Tests the {@link MedicalRecordRules#getEventForAddition(Party, Date, Entity, Party)} method.
     */
    @Test
    public void testGetEventForAdditionForWithCompletedEventOnSameDay() {
        Act event1 = createEvent(getDatetime("2013-11-21 10:00:00"), getDatetime("2013-11-21 11:00:00"), COMPLETED);
        Act event2 = createEvent(getDatetime("2013-11-21 12:00:05"), null, IN_PROGRESS);

        // no events, so a new event should be created
        checkGetEventForAddition(null, "2013-11-20 00:00:00", location);

        // timestamps closest to event1 should return event1
        checkGetEventForAddition(event1, "2013-11-21 00:00:00", location);
        checkGetEventForAddition(event1, "2013-11-21 09:00:00", location);
        checkGetEventForAddition(event1, "2013-11-21 10:00:00", location);
        checkGetEventForAddition(event1, "2013-11-21 11:00:00", location);

        // timestamps closest to event2 should return event2
        checkGetEventForAddition(event2, "2013-11-21 12:00:00", location);
        checkGetEventForAddition(event2, "2013-11-21 13:00:00", location);
        checkGetEventForAddition(event2, "2013-11-22 00:00:00", location);
        checkGetEventForAddition(event2, "2013-11-28 00:00:00", location);

        // over a week after event2, a new event should be created
        checkGetEventForAddition(null, "2013-11-29 00:00:00", location);

        // now make event2 a boarding event. It should now be returned
        Party customer = customerFactory.createCustomer();
        Entity cageType = schedulingFactory.createCageType();
        Entity schedule = schedulingFactory.newSchedule()
                .location(practiceFactory.createLocation())
                .cageType(cageType)
                .build();
        schedulingFactory.newAppointment().startTime("2013-11-21 11:50:00")
                .endTime("2013-11-30 17:00:00")
                .schedule(schedule)
                .appointmentType(schedulingFactory.createAppointmentType())
                .customer(customer)
                .patient(patient)
                .event(event2)
                .build();
        checkGetEventForAddition(event2, "2013-11-29 00:00:00", location);
    }

    /**
     * Tests the {@link MedicalRecordRules#getEventForAddition(Party, Date, Entity, Party)} method.
     */
    @Test
    public void testGetEventWithCompletedEventWithNoEndDate() {
        Act event1 = createEvent(getDatetime("2013-11-15 10:00:00"), null, COMPLETED);
        Act event2 = createEvent(getDatetime("2013-11-21 12:00:05"), null, IN_PROGRESS);

        // no events before the 15th, so a new event should be created
        checkGetEventForAddition(null, "2013-11-14 00:00:00", location);

        // event2 should be returned for all timestamps on the 15th
        checkGetEventForAddition(event1, "2013-11-15 00:00:00", location);
        checkGetEventForAddition(event1, "2013-11-15 23:59:59", location);

        // a new event should be returned from 16-20th
        checkGetEventForAddition(null, "2013-11-16 00:00:00", location);
        checkGetEventForAddition(null, "2013-11-20 23:59:59", location);

        // event2 should be returned for all timestamps on or after the 21st
        checkGetEventForAddition(event2, "2013-11-21 00:00:00", location);
        checkGetEventForAddition(event2, "2013-11-21 12:00:00", location);
        checkGetEventForAddition(event2, "2013-11-21 12:00:05", location);
        checkGetEventForAddition(event2, "2013-11-22 00:00:00", location);
        checkGetEventForAddition(event2, "2013-11-28 00:00:00", location);

        // over a week after event2, a new event should be created
        checkGetEventForAddition(null, "2013-11-29 13:00:00", location);

        // now make event2 a boarding event. It should now be returned
        Party customer = customerFactory.createCustomer();
        Entity cageType = schedulingFactory.createCageType();
        Entity schedule = schedulingFactory.newSchedule()
                .location(practiceFactory.createLocation())
                .cageType(cageType)
                .build();
        schedulingFactory.newAppointment()
                .startTime("2013-11-21 11:50:00")
                .endTime("2013-11-30 17:00:00")
                .schedule(schedule)
                .appointmentType(schedulingFactory.createAppointmentType())
                .customer(customer)
                .patient(patient)
                .event(event2)
                .build();

        checkGetEventForAddition(event2, "2013-11-29 13:00:00", location);
    }

    /**
     * Tests the {@link MedicalRecordRules#getLockableRecords()} method.
     */
    @Test
    public void testGetLockableRecords() {
        List<String> shortNames = Arrays.asList(rules.getLockableRecords());
        assertEquals(14, shortNames.size());
        assertTrue(shortNames.contains(PatientArchetypes.CLINICAL_ADDENDUM));
        assertTrue(shortNames.contains(PatientArchetypes.CLINICAL_NOTE));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_ATTACHMENT));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_ATTACHMENT_VERSION));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_FORM));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_IMAGE));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_IMAGE_VERSION));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_LETTER));
        assertTrue(shortNames.contains(PatientArchetypes.DOCUMENT_LETTER_VERSION));
        assertTrue(shortNames.contains(InvestigationArchetypes.PATIENT_INVESTIGATION));
        assertTrue(shortNames.contains(InvestigationArchetypes.PATIENT_INVESTIGATION_VERSION));
        assertTrue(shortNames.contains(PatientArchetypes.PATIENT_MEDICATION));
        assertTrue(shortNames.contains(PatientArchetypes.PATIENT_WEIGHT));
        assertTrue(shortNames.contains(PatientArchetypes.CLINICAL_LINK));
    }

    /**
     * Verifies that each of the lockable medical records have a startTime and status node.
     * If the status is hidden, it must be read-only.
     * <br/>
     * Note that in the original version of medical record locking, the startTime node was also read only.
     * However, this prevented the 1.8 behaviour of allowing practices to forward/back-date records when locking is not
     * enabled.
     */
    @Test
    public void testLockableRecordStartTimeAndStatusNodes() {
        IArchetypeService service = getArchetypeService();
        for (String shortName : rules.getLockableRecords()) {
            NodeDescriptor startTime = DescriptorHelper.getNode(shortName, "startTime", service);
            assertNotNull(shortName, startTime);

            NodeDescriptor status = DescriptorHelper.getNode(shortName, "status", service);
            assertNotNull(shortName, status);
            if (status.isHidden()) {
                assertTrue(shortName, status.isReadOnly());
            }
        }
    }

    /**
     * Tests the {@link MedicalRecordRules#getAttachment(String, Act)}.
     */
    @Test
    public void testGetAttachment() {
        Act event = createEvent();

        assertNull(rules.getAttachment("notes.pdf", event));
        DocumentAct act1 = createAttachment("2017-04-22 10:00:00", "notes.pdf");
        DocumentAct act2 = createAttachment("2017-04-22 11:00:00", "billing.pdf");
        patientFactory.updateVisit(event).addItems(act1, act2).build();

        assertEquals(act1, rules.getAttachment("notes.pdf", event));
        assertEquals(act2, rules.getAttachment("billing.pdf", event));

        DocumentAct act3 = createAttachment("2017-04-22 12:00:00", "notes.pdf");
        patientFactory.updateVisit(event).addItem(act3).build();

        assertEquals(act3, rules.getAttachment("notes.pdf", event));
    }

    /**
     * Tests the {@link MedicalRecordRules#getAttachment(String, Act, String, String)} method.
     */
    @Test
    public void testGetAttachmentWithIdentity() {
        Act event = createEvent();
        IMObjectBean bean = getBean(event);

        // Smart Flow Sheet supplies the same surgery UID for both anaesthetic and anaesthetic records reports.
        String archetype = "actIdentity.smartflowsheet";
        String identity1 = UUID.randomUUID().toString();
        String identity2 = UUID.randomUUID().toString();
        assertNull(rules.getAttachment("anaesthetic.pdf", event, archetype, identity1));

        DocumentAct act1a = createAttachment("2017-04-22 10:00:00", "anaesthetic.pdf", identity1);
        DocumentAct act1b = createAttachment("2017-04-22 10:00:00", "anaesthetic records.pdf", identity1);
        DocumentAct act2 = createAttachment("2017-04-22 11:00:00", "anaesthetic.pdf", identity2);

        bean.addTarget("items", act1a, "event");
        bean.addTarget("items", act1b, "event");
        bean.addTarget("items", act2, "event");
        bean.save();

        assertEquals(act1a, rules.getAttachment("anaesthetic.pdf", event, archetype, identity1));
        assertEquals(act1b, rules.getAttachment("anaesthetic records.pdf", event, archetype, identity1));
        assertEquals(act2, rules.getAttachment("anaesthetic.pdf", event, archetype, identity2));
    }

    /**
     * Helper to create an <em>act.patientClinicalEvent</em>.
     *
     * @return a new act
     */
    protected Act createEvent() {
        return patientFactory.newVisit()
                .patient(patient)
                .clinician(clinician)
                .location(location)
                .build();
    }

    /**
     * Helper to create an <em>act.patientClinicalEvent</em>.
     *
     * @param startTime the start time. May be {@code null}
     * @param endTime   the end time. May be {@code null}
     * @param status    the event status
     * @return a new act
     */
    protected Act createEvent(Date startTime, Date endTime, String status) {
        return newEvent(startTime, endTime)
                .status(status)
                .build();
    }

    /**
     * Helper to create an <em>act.patientClinicalEvent</em>.
     *
     * @param startTime the start time. May be {@code null}
     * @param endTime   the end time. May be {@code null}
     * @return a new act
     */
    protected Act createEvent(Date startTime, Date endTime) {
        return newEvent(startTime, endTime)
                .build();
    }

    /**
     * Helper to create an <em>act.patientClinicalEvent</em>.
     *
     * @param startTime the start time
     * @return a new act
     */
    protected Act createEvent(Date startTime) {
        return createEvent(startTime, null);
    }

    /**
     * Helper to create an <em>act.patientClinicalProblem</em>.
     *
     * @return a new act
     */
    protected Act createProblem() {
        return patientFactory.newProblem()
                .reason("HEART_MURMUR")
                .patient(patient)
                .clinician(clinician)
                .build();
    }

    /**
     * Helper to create an <em>act.patientMedication</em> for a patient.
     *
     * @param patient the patient
     * @return a new act
     */
    protected Act createMedication(Party patient) {
        return patientFactory.newMedication()
                .patient(patient)
                .product(productFactory.createMedication())
                .build();
    }

    /**
     * Helper to create an <em>act.patientClinicalNote</em>.
     *
     * @return a new act
     */
    protected Act createNote() {
        return patientFactory.createNote(patient, "a note");
    }

    /**
     * Helper to create an <em>act.patientClinicalAddendum</em>.
     *
     * @return a new act
     */
    protected Act createAddendum() {
        return patientFactory.createAddendum(patient, "an addendum");
    }

    /**
     * Helper to create an <em>act.patientDocumentAttachment</em>.
     *
     * @param datetime the date/time of the act
     * @param filename the file name
     * @return the attachment
     */
    private DocumentAct createAttachment(String datetime, String filename) {
        return patientFactory.newAttachment()
                .startTime(datetime)
                .patient(patient)
                .document(documentFactory.createPDF(filename))
                .build();
    }

    /**
     * Helper to create an <em>act.patientDocumentAttachment</em>.
     *
     * @param datetime the date/time of the act
     * @param filename the file name
     * @param identity the SmartFlow Sheet identity
     * @return the attachment
     */
    private DocumentAct createAttachment(String datetime, String filename, String identity) {
        return patientFactory.newAttachment()
                .startTime(datetime)
                .patient(patient)
                .addIdentity("actIdentity.smartflowsheet", identity)
                .document(documentFactory.createPDF(filename))
                .build();
    }

    /**
     * Returns a pre-populated visit builder.
     *
     * @param startTime the visit start
     * @param endTime   the visit end
     * @return the builder
     */
    private TestVisitBuilder newEvent(Date startTime, Date endTime) {
        return patientFactory.newVisit()
                .startTime(startTime)
                .endTime(endTime)
                .patient(patient)
                .location(location)
                .clinician(clinician);
    }

    /**
     * Verifies that the correct event is returned by {@link MedicalRecordRules#getEvent(Party)}.
     *
     * @param expected the expected event. May be {@code null}
     */
    private void checkGetEvent(Act expected) {
        Act event = rules.getEvent(patient);
        if (expected == null) {
            assertNull(event);
        } else {
            assertEquals(expected, event);
        }
    }

    /**
     * Verifies that the correct event is returned for a particular date by
     * {@link MedicalRecordRules#getEvent(Party, Date, Party)}
     *
     * @param date     the date
     * @param expected the expected event. May be {@code null}
     */
    private void checkGetEvent(Date date, Act expected) {
        Act event = rules.getEvent(patient, date, location);
        if (expected == null) {
            assertNull(event);
        } else {
            assertEquals(expected, event);
        }
    }

    /**
     * Verifies an event matches that expected.
     *
     * @param event     the event
     * @param startTime the expected start time
     * @param endTime   the expected end time. May be {@code null}
     * @param status    the expected status
     * @param location  the expected location. May be {@code null}
     */
    private void checkEvent(Act event, Date startTime, Date endTime, String status, Party location) {
        assertEquals(startTime, event.getActivityStartTime());
        assertEquals(endTime, event.getActivityEndTime());
        assertEquals(status, event.getStatus());
        IMObjectBean bean = getBean(event);
        assertEquals(patient, bean.getTarget("patient"));
        assertEquals(location, bean.getTarget("location"));
    }

    /**
     * Verifies that an event contains a set of acts.
     *
     * @param event the event
     * @param acts  the expected acts
     */
    private void checkContains(Act event, Act... acts) {
        List<Act> items = getActs(event);
        assertEquals(acts.length, items.size());
        for (Act act : acts) {
            boolean found = false;
            for (Act item : items) {
                if (item.equals(act)) {
                    found = true;
                    break;
                }
            }
            assertTrue(found);
        }
    }

    /**
     * Returns the items linked to an event.
     *
     * @param event the event
     * @return the items
     */
    private List<Act> getActs(Act event) {
        IMObjectBean bean = getBean(event);
        return bean.getTargets("items", Act.class);
    }

    /**
     * Verifies that the expected event is returned for a give date by
     * {@link MedicalRecordRules#getEventForAddition(Party, Date, Entity, Party)}.
     *
     * @param expected the expected event. If {@code null}, a new event is expected
     * @param date     the date
     */
    private void checkGetEventForAddition(Act expected, String date, Party location) {
        Date startTime = getDatetime(date);
        Act actual = rules.getEventForAddition(patient, startTime, null, location);
        if (expected != null) {
            assertEquals(expected, actual);
        } else {
            assertTrue(actual.isNew());
            assertEquals(startTime, actual.getActivityStartTime());
            assertEquals(IN_PROGRESS, actual.getStatus());
        }
    }

    /**
     * Helper to change the patient for an act.
     *
     * @param act     the act
     * @param patient the new patient
     */
    private void setPatient(Act act, Party patient) {
        IMObjectBean itemBean = getBean(act);
        itemBean.setTarget("patient", patient);
        itemBean.save();
    }
}
