/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.supplier.delivery;

import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.archetype.test.builder.supplier.TestSupplierActItemVerifier;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;

/**
 * Verifies  <em>act.supplierDeliveryItem</em> instances match that expected, for testing purposes.
 *
 * @author Tim Anderson
 */
public class TestDeliveryItemVerifier extends TestSupplierActItemVerifier<TestDeliveryItemVerifier> {

    /**
     * The expected order item.
     */
    private ValueStrategy orderItem = ValueStrategy.value(null);

    /**
     * The expected supplier invoice line id.
     */
    private ValueStrategy supplierInvoiceLineId = ValueStrategy.value(null);

    /**
     * Constructs a {@link TestDeliveryItemVerifier}.
     *
     * @param service the archetype service
     */
    public TestDeliveryItemVerifier(ArchetypeService service) {
        super(service);
    }

    /**
     * Sets the expected order item
     *
     * @param orderItem the order item
     * @return this
     */
    public TestDeliveryItemVerifier orderItem(FinancialAct orderItem) {
        return orderItem((orderItem != null) ? orderItem.getObjectReference() : null);
    }

    /**
     * Sets the expected order item
     *
     * @param orderItem the order item
     * @return this
     */
    public TestDeliveryItemVerifier orderItem(Reference orderItem) {
        this.orderItem = ValueStrategy.value(orderItem);
        return this;
    }

    /**
     * Sets the expected supplier invoice line identifier.
     *
     * @param supplierInvoiceLineId the supplier invoice line identifier
     * @return this
     */
    public TestDeliveryItemVerifier supplierInvoiceLineId(String supplierInvoiceLineId) {
        this.supplierInvoiceLineId = ValueStrategy.value(supplierInvoiceLineId);
        return this;
    }

    /**
     * Initialises this from an item.
     *
     * @param bean the bean wrapping the item
     */
    @Override
    protected void initialise(IMObjectBean bean) {
        super.initialise(bean);
        orderItem(bean.getTargetRef("order"));
    }

    /**
     * Verifies an object matches that expected.
     *
     * @param object the object to verify
     * @param bean   a bean wrapping the object
     */
    @Override
    protected void verify(FinancialAct object, IMObjectBean bean) {
        super.verify(object, bean);
        checkEquals(orderItem, bean.getTargetRef("order"));
        checkEquals(supplierInvoiceLineId, bean.getString("supplierInvoiceLineId"));
    }
}
