/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.supplier;

import org.openvpms.archetype.test.builder.act.AbstractTestActVerifier;
import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.math.BigDecimal;

/**
 * Verifies <em>act.supplierOrder</em>, <em>act.supplierDelivery</em> and <em>act.supplierReturn</em> instances
 * match that expected, for testing purposes.
 *
 * @author Tim Anderson
 */
public abstract class TestSupplierActVerifier<V extends TestSupplierActVerifier<V>>
        extends AbstractTestActVerifier<FinancialAct, V> {

    /**
     * The expected supplier.
     */
    private ValueStrategy supplier = ValueStrategy.value(null);

    /**
     * The expected location.
     */
    private ValueStrategy stockLocation = ValueStrategy.value(null);

    /**
     * The expected amount.
     */
    private ValueStrategy amount = ValueStrategy.value(null);

    /**
     * The expected tax.
     */
    private ValueStrategy tax = ValueStrategy.value(null);

    /**
     * Constructs a {@link TestSupplierActVerifier}.
     *
     * @param service the archetype service
     */
    public TestSupplierActVerifier(ArchetypeService service) {
        super(service);
    }

    /**
     * Initialises this from an act.
     *
     * @param act the act
     * @return this
     */
    public V initialise(FinancialAct act) {
        IMObjectBean bean = getBean(act);
        supplier(bean.getTargetRef("supplier"));
        stockLocation(bean.getTargetRef("stockLocation"));
        amount(bean.getBigDecimal("amount"));
        tax(bean.getBigDecimal("tax"));
        status(bean.getString("status"));
        return getThis();
    }

    /**
     * Sets the expected supplier.
     *
     * @param supplier the supplier
     * @return this
     */
    public V supplier(Party supplier) {
        return supplier((supplier != null) ? supplier.getObjectReference() : null);
    }

    /**
     * Sets the expected supplier.
     *
     * @param supplier the supplier
     * @return this
     */
    public V supplier(Reference supplier) {
        this.supplier = ValueStrategy.value(supplier);
        return getThis();
    }

    /**
     * Sets the stock location.
     *
     * @param stockLocation the stock location
     * @return this
     */
    public V stockLocation(Party stockLocation) {
        return stockLocation((stockLocation != null) ? stockLocation.getObjectReference() : null);
    }

    /**
     * Sets the expected stock location.
     *
     * @param stockLocation the expected stock location
     * @return this
     */
    public V stockLocation(Reference stockLocation) {
        this.stockLocation = ValueStrategy.value(stockLocation);
        return getThis();
    }

    /**
     * Sets the expected amount.
     *
     * @param amount the amount
     * @return this
     */
    public V amount(BigDecimal amount) {
        this.amount = ValueStrategy.value(amount);
        return getThis();
    }

    /**
     * Sets the expected tax.
     *
     * @param tax the tax
     * @return this
     */
    public V tax(BigDecimal tax) {
        this.tax = ValueStrategy.value(tax);
        return getThis();
    }

    /**
     * Verifies an object matches that expected.
     *
     * @param object the object to verify
     * @param bean   a bean wrapping the object
     */
    @Override
    protected void verify(FinancialAct object, IMObjectBean bean) {
        super.verify(object, bean);
        checkEquals(supplier, bean.getTargetRef("supplier"));
        checkEquals(stockLocation, bean.getTargetRef("stockLocation"));
        checkEquals(amount, bean.getBigDecimal("amount"));
        checkEquals(tax, bean.getBigDecimal("tax"));
    }
}
