/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.paymentprocessor;

import org.openvpms.archetype.test.builder.act.AbstractTestActBuilder;
import org.openvpms.archetype.test.builder.customer.account.TestPaymentProcessorItemBuilder.TransactionMode;
import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.math.BigDecimal;
import java.util.Set;

/**
 * Builder for <em>act.paymentProcessor*</em> instances, for testing purposes.
 *
 * @author Tim Anderson
 */
public abstract class TestPaymentProcessorTransactionBuilder<B extends TestPaymentProcessorTransactionBuilder<B>>
        extends AbstractTestActBuilder<FinancialAct, B> {

    /**
     * The customer.
     */
    private Party customer;

    /**
     * The transaction amount,
     */
    private BigDecimal amount;

    /**
     * The payment processor.
     */
    private Entity paymentProcessor;

    /**
     * The transaction mode.
     */
    private ValueStrategy transactionMode = ValueStrategy.unset();

    /**
     * The practice location.
     */
    private Entity location;

    /**
     * The email address.
     */
    private ValueStrategy email = ValueStrategy.unset();

    /**
     * Constructs an {@link TestPaymentProcessorTransactionBuilder}.
     *
     * @param archetype the archetype to build
     * @param service   the archetype service
     */
    public TestPaymentProcessorTransactionBuilder(String archetype, ArchetypeService service) {
        super(archetype, FinancialAct.class, service);
        transactionMode(TransactionMode.LINK);
    }

    /**
     * Sets the customer.
     *
     * @param customer the customer
     */
    public B customer(Party customer) {
        this.customer = customer;
        return getThis();
    }

    /**
     * Sets the amount.
     *
     * @param amount the amount
     * @return this
     */
    public B amount(int amount) {
        return amount(BigDecimal.valueOf(amount));
    }

    /**
     * Sets the amount.
     *
     * @param amount the amount
     * @return this
     */
    public B amount(BigDecimal amount) {
        this.amount = amount;
        return getThis();
    }

    /**
     * Sets the payment processor.
     *
     * @param paymentProcessor the payment processor
     * @return this
     */
    public B paymentProcessor(Entity paymentProcessor) {
        this.paymentProcessor = paymentProcessor;
        return getThis();
    }

    /**
     * Sets the transaction mode.
     *
     * @param transactionMode the transaction mode
     * @return this
     */
    public B transactionMode(TransactionMode transactionMode) {
        this.transactionMode = ValueStrategy.value(transactionMode.toString());
        return getThis();
    }

    /**
     * Sets the practice location.
     *
     * @param location the practice location
     * @return this
     */
    public B location(Party location) {
        this.location = location;
        return getThis();
    }

    /**
     * Sets the email address.
     *
     * @param email the email address
     * @return this
     */
    public B email(String email) {
        this.email = ValueStrategy.value(email);
        return getThis();
    }

    /**
     * Builds the object.
     *
     * @param object   the object
     * @param bean     a bean wrapping the object
     * @param toSave   objects to save, if the object is to be saved
     * @param toRemove the objects to remove
     */
    @Override
    protected void build(FinancialAct object, IMObjectBean bean, Set<IMObject> toSave, Set<Reference> toRemove) {
        super.build(object, bean, toSave, toRemove);
        if (customer != null) {
            bean.setTarget("customer", customer);
        }
        if (amount != null) {
            bean.setValue("amount", amount);
        }
        if (paymentProcessor != null) {
            bean.setTarget("paymentProcessor", paymentProcessor);
        }
        transactionMode.setValue(bean, "transactionMode");
        if (location != null) {
            bean.setTarget("location", location);
        }
        email.setValue(bean, "email");
    }
}
