/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.object;

import org.openvpms.component.model.object.IMObject;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static org.junit.Assert.assertEquals;

/**
 * Tracks objects built by {@link AbstractTestIMObjectBuilder}.
 *
 * @author Tim Anderson
 */
public class BuiltObjects {

    private final IMObject root;

    private final Set<IMObject> objects;

    public BuiltObjects(IMObject root, Set<IMObject> objects) {
        this.root = root;
        this.objects = objects;
    }

    /**
     * Returns the root object.
     *
     * @return the root object
     */
    public IMObject getRoot() {
        return root;
    }

    /**
     * Returns the root object, cast to the specified type.
     *
     * @param type the type
     * @return the root object
     */
    public <T extends IMObject> T getRoot(Class<T> type) {
        return type.cast(root);
    }

    /**
     * Returns all objects matching the specified archetype.
     *
     * @param archetype the archetype
     * @return the objects matching the archetype
     */
    public List<IMObject> getObjects(String archetype) {
        return objects.stream().filter(object -> object.isA(archetype)).collect(Collectors.toList());
    }

    /**
     * Returns all objects matching the specified archetype.
     *
     * @param archetype the archetype
     * @param type      the expected type
     * @return the objects matching the archetype
     */
    public <T extends IMObject> List<T> getObjects(String archetype, Class<T> type) {
        return getObjects(archetype).stream().map(type::cast).collect(Collectors.toList());
    }

    /**
     * Assert that there is a single built object of the specified archetype.
     *
     * @param archetype the object archetype
     * @param type the object type
     * @return the corresponding object
     */
    public <T extends IMObject> T getObject(String archetype, Class<T> type) {
        List<T> matches = getObjects(archetype, type);
        assertEquals(1, matches.size());
        return matches.get(0);
    }

}
