/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.customer.account;

import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorArchetypes;
import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.archetype.test.builder.paymentprocessor.TestPaymentProcessorTransactionBuilder;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Builds <em>act.customerAccountPaymentPP</em> and <em>act.customerAccountRefundPP</em>instances, for testing
 * purposes.
 *
 * @author Tim Anderson
 */
public abstract class TestPaymentProcessorItemBuilder<
        P extends TestPaymentRefundBuilder<P>, B extends TestPaymentProcessorItemBuilder<P, B>>
        extends TestPaymentRefundItemBuilder<P, B> {

    public enum TransactionMode {
        LINK,
        STORED_CARD
    }

    /**
     * The transactions to add.
     */
    private final List<FinancialAct> transactions = new ArrayList<>();

    /**
     * The transactions to create. Each element indicates the transaction status.
     */
    private final List<String> transactionsToCreate = new ArrayList<>();

    /**
     * The payment processor.
     */
    private Entity paymentProcessor;

    /**
     * The transaction mode.
     */
    private ValueStrategy transactionMode = ValueStrategy.unset();

    /**
     * The built transactions, available after building.
     */
    private List<FinancialAct> builtTransactions = new ArrayList<>();

    /**
     * Constructs an {@link TestPaymentProcessorItemBuilder}.
     *
     * @param parent    the parent builder
     * @param archetype the archetype to create
     * @param service   the archetype service
     */
    protected TestPaymentProcessorItemBuilder(P parent, String archetype, ArchetypeService service) {
        super(parent, archetype, service);
        transactionMode(TransactionMode.LINK);
    }

    /**
     * Constructs an {@link TestPaymentProcessorItemBuilder}.
     *
     * @param object  the object to update
     * @param parent  the parent builder
     * @param service the archetype service
     */
    public TestPaymentProcessorItemBuilder(FinancialAct object, P parent, ArchetypeService service) {
        super(object, parent, service);
        paymentProcessor = getBean(object).getTarget("paymentProcessor", Entity.class);
    }

    /**
     * Sets the payment processor.
     *
     * @param paymentProcessor the payment processor
     * @return this
     */
    public B processor(Entity paymentProcessor) {
        this.paymentProcessor = paymentProcessor;
        return getThis();
    }

    /**
     * Sets the transaction mode.
     *
     * @param transactionMode the transaction mode
     * @return this
     */
    public B transactionMode(TransactionMode transactionMode) {
        this.transactionMode = ValueStrategy.value(transactionMode.toString());
        return getThis();
    }

    /**
     * Adds a transaction with the specified status.
     *
     * @param status the status
     */
    public B addTransaction(String status) {
        if (getParent() == null) {
            // need the customer and location from the parent
            throw new IllegalStateException("Parent builder is required by addTransaction(status)");
        }
        transactionsToCreate.add(status);
        return getThis();
    }

    /**
     * Adds a payment processor transaction.
     *
     * @param transaction the transaction
     */
    public B addTransaction(FinancialAct transaction) {
        transactions.add(transaction);
        return getThis();
    }

    /**
     * Returns the built transactions.
     *
     * @return the built transactions
     */
    public List<FinancialAct> getTransactions() {
        return builtTransactions;
    }

    /**
     * Builds the object.
     *
     * @param object   the object
     * @param bean     a bean wrapping the object
     * @param toSave   objects to save, if the object is to be saved
     * @param toRemove the objects to remove
     */
    @Override
    protected void build(FinancialAct object, IMObjectBean bean, Set<IMObject> toSave, Set<Reference> toRemove) {
        super.build(object, bean, toSave, toRemove);
        bean.setTarget("paymentProcessor", paymentProcessor);
        transactionMode.setValue(bean, "transactionMode");
        P parent = getParent();
        if (!transactionsToCreate.isEmpty()) {
            for (String status : transactionsToCreate) {
                TestPaymentProcessorTransactionBuilder<?> builder = createTransactionBuilder();
                Party customer = parent.getCustomer();
                if (customer == null) {
                    throw new IllegalStateException("Parent customer has not been set");
                }
                Party location = parent.getLocation();
                if (location == null) {
                    throw new IllegalStateException("Parent location has not been set");
                }
                FinancialAct act = builder.customer(customer)
                        .amount(object.getTotal())
                        .status(status)
                        .paymentProcessor(paymentProcessor)
                        .location(location)
                        .build(false);
                transactions.add(act);
            }
        }
        if (!transactions.isEmpty()) {
            int sequence = getNextSequence(bean, "transactions");
            for (FinancialAct act : transactions) {
                ActRelationship relationship = (ActRelationship) bean.addTarget("transactions", act);
                act.addActRelationship(relationship);
                relationship.setSequence(sequence++);
                toSave.add(act);

                if (act.getIdentities(PaymentProcessorArchetypes.PARENT_ID).isEmpty() && parent != null) {
                    parent.addPostBuildAction((builtObject, builtBean) -> {
                        // need to add the identity after the parent is saved, to ensure it has a persistent id
                        if (builtObject.isNew()) {
                            throw new IllegalStateException(builtObject.getArchetype() + " must be saved");
                        }
                        act.addIdentity(createActIdentity(PaymentProcessorArchetypes.PARENT_ID,
                                                          Long.toString(builtObject.getId())));
                        getService().save(act);
                    });
                }
            }
            builtTransactions = new ArrayList<>(transactions);
            transactions.clear();
        }
    }

    /**
     * Returns a builder to create a payment processor transaction.
     *
     * @return a new builder
     */
    protected abstract TestPaymentProcessorTransactionBuilder<?> createTransactionBuilder();
}
