/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.workflow;

import org.joda.time.Period;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.rules.workflow.roster.RosterService;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.customer.TestCustomerFactory;
import org.openvpms.archetype.test.builder.patient.TestPatientFactory;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.archetype.test.builder.scheduling.TestAppointmentBuilder;
import org.openvpms.archetype.test.builder.scheduling.TestScheduleBuilder;
import org.openvpms.archetype.test.builder.scheduling.TestSchedulingFactory;
import org.openvpms.archetype.test.builder.user.TestUserFactory;
import org.openvpms.component.business.service.cache.BasicEhcacheManager;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.user.User;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;
import java.util.Iterator;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.openvpms.archetype.test.TestHelper.getDate;
import static org.openvpms.archetype.test.TestHelper.getDatetime;

/**
 * Tests the {@link FreeSlotQuery} class.
 *
 * @author Tim Anderson
 */
public class FreeSlotQueryTestCase extends ArchetypeServiceTest {

    /**
     * The appointment rules.
     */
    @Autowired
    private AppointmentRules appointmentRules;

    /**
     * The customer factory.
     */
    @Autowired
    private TestCustomerFactory customerFactory;

    /**
     * The patient factory.
     */
    @Autowired
    private TestPatientFactory patientFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * The scheduling factory.
     */
    @Autowired
    private TestSchedulingFactory schedulingFactory;

    /**
     * The user factory.
     */
    @Autowired
    private TestUserFactory userFactory;

    /**
     * The appointment service.
     */
    private AppointmentService appointmentService;

    /**
     * The roster service.
     */
    private RosterService rosterService;

    /**
     * The practice location.
     */
    private Party location;

    /**
     * Sets up the test.
     */
    @Before
    public void setUp() {
        location = practiceFactory.createLocation();
        appointmentService = new AppointmentService(getArchetypeService(), getLookupService(),
                                                    new BasicEhcacheManager(30));
        rosterService = new RosterService(getArchetypeService(), new BasicEhcacheManager(30));
    }

    /**
     * Cleans up after the test.
     *
     * @throws Exception for any error
     */
    @After
    public void tearDown() throws Exception {
        appointmentService.destroy();
        rosterService.destroy();
    }

    /**
     * Tests finding free slots where the schedule doesn't define start and end times.
     */
    @Test
    public void testFindFreeSlotsForSingleSchedule() {
        checkFindFreeSlotsForSingleSchedule(true);
        checkFindFreeSlotsForSingleSchedule(false);
    }

    /**
     * Tests finding free slots for multiple schedules, where the schedules don't define start and end times.
     */
    @Test
    public void testFindFreeSlotsForMultipleSchedules() {
        checkFindFreeSlotsForMultipleSchedules(true);
        checkFindFreeSlotsForMultipleSchedules(false);
    }

    /**
     * Verifies that a free slot with the same length as the query range is returned if there are no appointments.
     */
    @Test
    public void testFindFreeSlotsForEmptySchedule() {
        checkFindFreeSlotsForEmptySchedule(true);
        checkFindFreeSlotsForEmptySchedule(false);
    }

    /**
     * Tests finding free slots when there is a single appointment during the date range.
     */
    @Test
    public void testFindFreeSlotsForAppointmentDuringDateRange() {
        checkFindFreeSlotsForAppointmentDuringDateRange(true);
        checkFindFreeSlotsForAppointmentDuringDateRange(false);
    }

    /**
     * Tests finding free slots when there is a single appointment on the start of the date range.
     */
    @Test
    public void testFindFreeSlotForAppointmentAtStartOfDateRange() {
        checkFindFreeSlotForAppointmentAtStartOfDateRange(true);
        checkFindFreeSlotForAppointmentAtStartOfDateRange(false);
    }

    /**
     * Tests finding free slots when there is a single appointment overlapping the start of the date range.
     */
    @Test
    public void testFindFreeSlotForAppointmentOverlappingStart() {
        checkFindFreeSlotForAppointmentOverlappingStart(true);
        checkFindFreeSlotForAppointmentOverlappingStart(false);
    }

    /**
     * Verifies that free slots are handled correctly if a schedule has a single appointment at the start, during
     * or at the end of the date range.
     */
    @Test
    public void testFindFreeSlotsForSingleAppointment() {
        checkFindFreeSlotsForSingleAppointment(true);
        checkFindFreeSlotsForSingleAppointment(false);
    }

    /**
     * Verifies that duplicate appointments don't cause duplicate free slots to be reported.
     */
    @Test
    public void testDuplicateAppointments() {
        checkDuplicateAppointments(true);
        checkDuplicateAppointments(false);
    }

    /**
     * Verifies that overlapping appointments are handled.
     */
    @Test
    public void testOverlappingAppointments() {
        checkOverlappingAppointments(true);
        checkOverlappingAppointments(false);
    }

    /**
     * Verifies that specifying a minimum slot size filters out slots too small.
     */
    @Test
    public void testMinSlotSize() {
        checkMinSlotSize(true);
        checkMinSlotSize(false);
    }

    /**
     * Verifies that when a schedule has start and end times, free slots will be adjusted.
     */
    @Test
    public void testFindFreeSlotsForLimitedScheduleTimes() {
        checkFindFreeSlotsForLimitedScheduleTimes(true);
        checkFindFreeSlotsForLimitedScheduleTimes(false);
    }

    /**
     * Verifies that when a {@link FreeSlotQuery#setFromTime(Period)} is specified, only free slots after that
     * time are returned.
     */
    @Test
    public void testFindFreeSlotsWithFromTimeRange() {
        checkFindFreeSlotsWithFromTimeRange(true);
        checkFindFreeSlotsWithFromTimeRange(false);
    }

    /**
     * Verifies that when a {@link FreeSlotQuery#setToTime(Period)} is specified, only free slots before that
     * time are returned.
     */
    @Test
    public void testFindFreeSlotsWithToTimeRange() {
        checkFindFreeSlotsWithToTimeRange(true);
        checkFindFreeSlotsWithToTimeRange(false);
    }

    /**
     * Verifies that when both a {@link FreeSlotQuery#setFromTime(Period)} and
     * {@link FreeSlotQuery#setToTime(Period)} is specified, only free slots between those times are returned.
     */
    @Test
    public void testFindFreeSlotsWithFromToTimeRange() {
        checkFindFreeSlotsWithFromToTimeRange(true);
        checkFindFreeSlotsWithFromToTimeRange(false);
    }

    /**
     * Verifies that when a schedule has start and end times, free slots are split over those times.
     */
    @Test
    public void testFreeSlotSplitsOverScheduleTimes() {
        checkFreeSlotSplitsOverScheduleTimes(true);
        checkFreeSlotSplitsOverScheduleTimes(false);
    }

    /**
     * Verifies that slots are returned when a schedule starts at 0:00 and ends at 24:00.
     */
    @Test
    public void testFindFreeSlotFor24HourSchedule() {
        checkFindFreeSlotFor24HourSchedule(true);
        checkFindFreeSlotFor24HourSchedule(false);
    }

    /**
     * Verifies that slots are returned when a schedule starts at 0:00.
     */
    @Test
    public void testFindFreeSlotFor0HourStart() {
        checkFindFreeSlotFor0HourStart(true);
        checkFindFreeSlotFor0HourStart(false);
    }

    /**
     * Verifies that slots are returned when a schedule ends at 24:00.
     */
    @Test
    public void testFindFreeSlotFor24HourEnd() {
        checkFindFreeSlotFor24HourEnd(true);
        checkFindFreeSlotFor24HourEnd(false);
    }

    /**
     * Verifies that slots can be filtered by cage type.
     */
    @Test
    public void testFindFreeSlotForCageType() {
        Entity cageType1 = schedulingFactory.createCageType();
        Entity cageType2 = schedulingFactory.createCageType();
        Entity schedule = createSchedule("09:00", "24:00", cageType1);
        FreeSlotQuery query = createQuery("2014-01-01", "2014-01-03", schedule);
        query.setCageType(cageType1);

        Iterator<Slot> iterator1 = createIterator(query);
        checkSlot(iterator1, schedule, "2014-01-01 09:00:00", "2014-01-02 00:00:00");
        checkSlot(iterator1, schedule, "2014-01-02 09:00:00", "2014-01-03 00:00:00");

        query.setCageType(cageType2);
        Iterator<Slot> iterator2 = createIterator(query);
        assertFalse(iterator2.hasNext());

        query.setCageType(null);
        Iterator<Slot> iterator3 = createIterator(query);
        checkSlot(iterator3, schedule, "2014-01-01 09:00:00", "2014-01-02 00:00:00");
        checkSlot(iterator3, schedule, "2014-01-02 09:00:00", "2014-01-03 00:00:00");
    }

    /**
     * Verifies that no slots are returned for a clinician if they are not rostered on.
     */
    @Test
    public void testUnrosteredClinician() {
        Entity schedule1 = createSchedule(null, null, true);
        createAppointment("2019-04-01 09:00:00", "2019-04-01 09:15:00", schedule1);
        createAppointment("2019-04-01 10:00:00", "2019-04-01 10:15:00", schedule1);

        FreeSlotQuery query = createQuery("2019-04-01", "2019-04-02", schedule1);
        query.setClinician(userFactory.createClinician());
        Iterator<Slot> iterator = createIterator(query);
        assertFalse(iterator.hasNext());
    }

    /**
     * Verifies that if a clinician is rostered on, slots are returned.
     */
    @Test
    public void testRosteredClinician() {
        User clinician = userFactory.createClinician();
        Entity schedule1 = createSchedule(null, null, true);
        Entity schedule2 = createSchedule(null, null, true);
        Entity area = schedulingFactory.createRosterArea(location, schedule1, schedule2);
        Act shift = ScheduleTestHelper.createRosterEvent(getDatetime("2019-04-01 09:00:00"),
                                                         getDatetime("2019-04-01 17:00:00"),
                                                         clinician, area, location);
        save(shift);

        FreeSlotQuery query = createQuery("2019-04-01", "2019-04-02", schedule1);
        query.setClinician(clinician);

        Iterator<Slot> iterator1 = createIterator(query);
        checkSlot(iterator1, schedule1, "2019-04-01 09:00:00", "2019-04-01 17:00:00");

        // now create some appointments
        createAppointment("2019-04-01 09:00:00", "2019-04-01 09:15:00", schedule1);
        createAppointment("2019-04-01 10:00:00", "2019-04-01 10:15:00", schedule1);
        createAppointment("2019-04-01 10:15:00", "2019-04-01 10:45:00", schedule2, clinician);
        createAppointment("2019-04-01 11:00:00", "2019-04-01 11:30:00", schedule2, clinician);

        Iterator<Slot> iterator2 = createIterator(query);
        checkSlot(iterator2, schedule1, "2019-04-01 09:15:00", "2019-04-01 10:00:00");
        checkSlot(iterator2, schedule1, "2019-04-01 10:45:00", "2019-04-01 11:00:00");
        checkSlot(iterator2, schedule1, "2019-04-01 11:30:00", "2019-04-01 17:00:00");

        // create a new appointment at the end of the shift
        createAppointment("2019-04-01 15:00:00", "2019-04-01 16:00:00", schedule2, clinician);

        Iterator<Slot> iterator3 = createIterator(query);
        checkSlot(iterator3, schedule1, "2019-04-01 09:15:00", "2019-04-01 10:00:00");
        checkSlot(iterator3, schedule1, "2019-04-01 10:45:00", "2019-04-01 11:00:00");
        checkSlot(iterator3, schedule1, "2019-04-01 11:30:00", "2019-04-01 15:00:00");
        checkSlot(iterator3, schedule1, "2019-04-01 16:00:00", "2019-04-01 17:00:00");

        // create an appointment overlapping the end of the shift
        createAppointment("2019-04-01 16:00:00", "2019-04-01 17:30:00", schedule2, clinician);

        Iterator<Slot> iterator4 = createIterator(query);
        checkSlot(iterator4, schedule1, "2019-04-01 09:15:00", "2019-04-01 10:00:00");
        checkSlot(iterator4, schedule1, "2019-04-01 10:45:00", "2019-04-01 11:00:00");
        checkSlot(iterator4, schedule1, "2019-04-01 11:30:00", "2019-04-01 15:00:00");

        // now look from the perspective of schedule2
        query.setSchedules(schedule2);
        Iterator<Slot> iterator5 = createIterator(query);
        checkSlot(iterator5, schedule2, "2019-04-01 09:00:00", "2019-04-01 10:15:00");
        checkSlot(iterator5, schedule2, "2019-04-01 10:45:00", "2019-04-01 11:00:00");
        checkSlot(iterator5, schedule2, "2019-04-01 11:30:00", "2019-04-01 15:00:00");
    }

    /**
     * Tests finding free slots where the schedule doesn't define start and end times.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForSingleSchedule(boolean maxDuration) {
        Entity schedule1 = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule1);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule1);

        Iterator<Slot> iterator = createIterator("2014-01-01", "2014-01-02", schedule1);
        checkSlot(iterator, schedule1, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(iterator, schedule1, "2014-01-01 09:15:00", "2014-01-01 10:00:00");
        checkSlot(iterator, schedule1, "2014-01-01 10:15:00", "2014-01-02 00:00:00");
        assertFalse(iterator.hasNext());
    }

    /**
     * Tests finding free slots for multiple schedules, where the schedules don't define start and end times.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForMultipleSchedules(boolean maxDuration) {
        Entity schedule1 = createSchedule(null, null, maxDuration);
        Entity schedule2 = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule1);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule1);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:30:00", schedule2);
        createAppointment("2014-01-01 09:45:00", "2014-01-01 10:30:00", schedule2);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule1, schedule2);
        checkSlot(query, schedule1, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(query, schedule2, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(query, schedule1, "2014-01-01 09:15:00", "2014-01-01 10:00:00");
        checkSlot(query, schedule2, "2014-01-01 09:30:00", "2014-01-01 09:45:00");
        checkSlot(query, schedule1, "2014-01-01 10:15:00", "2014-01-02 00:00:00");
        checkSlot(query, schedule2, "2014-01-01 10:30:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that a free slot with the same length as the query range is returned if there are no appointments.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForEmptySchedule(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 00:00:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Tests finding free slots when there is a single appointment during the date range.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForAppointmentDuringDateRange(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(query, schedule, "2014-01-01 09:15:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Tests finding free slots when there is a single appointment on the start of the date range.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotForAppointmentAtStartOfDateRange(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 00:00:00", "2014-01-01 09:00:00", schedule);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 09:00:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Tests finding free slots when there is a single appointment overlapping the start of the date range.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotForAppointmentOverlappingStart(boolean maxDuration) {
        // test an appointment overlapping the start of the date range
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 08:00:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that free slots are handled correctly if a schedule has a single appointment at the start, during
     * or at the end of the date range.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForSingleAppointment(boolean maxDuration) {
        // test an appointment at the end of the date range
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 17:00:00", "2014-01-02 00:00:00", schedule);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 00:00:00", "2014-01-01 17:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that duplicate appointments don't cause duplicate free slots to be reported.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkDuplicateAppointments(boolean maxDuration) {
        Entity schedule1 = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule1);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule1);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule1);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule1);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule1);
        checkSlot(query, schedule1, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(query, schedule1, "2014-01-01 09:15:00", "2014-01-01 10:00:00");
        checkSlot(query, schedule1, "2014-01-01 10:15:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that overlapping appointments are handled.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkOverlappingAppointments(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:30:00", schedule);
        createAppointment("2014-01-01 09:45:00", "2014-01-01 10:15:00", schedule);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:30:00", schedule);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-02", schedule);
        checkSlot(query, schedule, "2014-01-01 00:00:00", "2014-01-01 09:00:00");
        checkSlot(query, schedule, "2014-01-01 09:30:00", "2014-01-01 09:45:00");
        checkSlot(query, schedule, "2014-01-01 10:30:00", "2014-01-02 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that specifying a minimum slot size filters out slots too small.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkMinSlotSize(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:45:00", schedule);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule);
        createAppointment("2014-01-01 11:00:00", "2014-01-01 11:15:00", schedule);

        FreeSlotQuery query = createQuery("2014-01-01", "2014-01-02", schedule);
        query.setMinSlotSize(30, DateUnits.MINUTES);

        Iterator<Slot> iterator = query.query();
        checkSlot(iterator, schedule, "2014-01-01 08:00:00", "2014-01-01 09:00:00");
        checkSlot(iterator, schedule, "2014-01-01 10:15:00", "2014-01-01 11:00:00");
        checkSlot(iterator, schedule, "2014-01-01 11:15:00", "2014-01-02 00:00:00");
        assertFalse(iterator.hasNext());
    }

    /**
     * Verifies that when a schedule has start and end times, free slots will be adjusted.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsForLimitedScheduleTimes(boolean maxDuration) {
        Entity schedule = createSchedule("09:00", "17:00", maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        createAppointment("2014-01-01 09:30:00", "2014-01-01 09:45:00", schedule);
        createAppointment("2014-01-02 09:00:00", "2014-01-02 10:00:00", schedule);

        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query, schedule, "2014-01-01 09:00:00", "2014-01-01 09:30:00");
        checkSlot(query, schedule, "2014-01-01 09:45:00", "2014-01-01 17:00:00");
        checkSlot(query, schedule, "2014-01-02 10:00:00", "2014-01-02 17:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that when a {@link FreeSlotQuery#setFromTime(Period)} is specified, only free slots after that
     * time are returned.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsWithFromTimeRange(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule);
        createAppointment("2014-01-01 10:30:00", "2014-01-01 11:00:00", schedule);

        FreeSlotQuery query = createQuery("2014-01-01", "2014-01-02", schedule);
        query.setFromTime(getTime("09:30"));
        Iterator<Slot> iterator = createIterator(query);
        checkSlot(iterator, schedule, "2014-01-01 09:30:00", "2014-01-01 10:00:00");
        checkSlot(iterator, schedule, "2014-01-01 10:15:00", "2014-01-01 10:30:00");
        checkSlot(iterator, schedule, "2014-01-01 11:00:00", "2014-01-02 00:00:00");
        assertFalse(iterator.hasNext());
    }

    /**
     * Verifies that when a {@link FreeSlotQuery#setToTime(Period)} is specified, only free slots before that
     * time are returned.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsWithToTimeRange(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule);
        createAppointment("2014-01-01 10:30:00", "2014-01-01 11:00:00", schedule);

        FreeSlotQuery query = createQuery("2014-01-01", "2014-01-02", schedule);
        query.setToTime(getTime("09:30"));
        Iterator<Slot> iterator = createIterator(query);
        checkSlot(iterator, schedule, "2014-01-01 08:00:00", "2014-01-01 09:00:00");
        checkSlot(iterator, schedule, "2014-01-01 09:15:00", "2014-01-01 09:30:00");
        assertFalse(iterator.hasNext());
    }

    /**
     * Verifies that when both a {@link FreeSlotQuery#setFromTime(Period)} and
     * {@link FreeSlotQuery#setToTime(Period)} is specified, only free slots between those times are returned.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotsWithFromToTimeRange(boolean maxDuration) {
        Entity schedule = createSchedule(null, null, maxDuration);
        createAppointment("2013-12-31 09:00:00", "2014-01-01 08:00:00", schedule);
        createAppointment("2014-01-01 09:00:00", "2014-01-01 09:15:00", schedule);
        createAppointment("2014-01-01 10:00:00", "2014-01-01 10:15:00", schedule);
        createAppointment("2014-01-01 10:30:00", "2014-01-01 11:00:00", schedule);

        FreeSlotQuery query = createQuery("2014-01-01", "2014-01-02", schedule);
        query.setFromTime(getTime("9:00"));
        query.setToTime(getTime("10:00"));
        query.setSchedules(schedule);
        Iterator<Slot> iterator = createIterator(query);
        checkSlot(iterator, schedule, "2014-01-01 09:15:00", "2014-01-01 10:00:00");
        assertFalse(iterator.hasNext());
    }

    /**
     * Verifies that when a schedule has start and end times, free slots are split over those times.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFreeSlotSplitsOverScheduleTimes(boolean maxDuration) {
        Entity schedule = createSchedule("09:00", "17:00", maxDuration);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query, schedule, "2014-01-01 09:00:00", "2014-01-01 17:00:00");
        checkSlot(query, schedule, "2014-01-02 09:00:00", "2014-01-02 17:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that slots are returned when a schedule starts at 0:00 and ends at 24:00.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotFor24HourSchedule(boolean maxDuration) {
        Entity schedule = createSchedule("00:00", "24:00", maxDuration);
        Iterator<Slot> query1 = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query1, schedule, "2014-01-01 00:00:00", "2014-01-03 00:00:00");
        assertFalse(query1.hasNext());

        createAppointment(getDate("2014-01-01"), getDate("2014-01-02"), schedule);
        Iterator<Slot> query2 = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query2, schedule, "2014-01-02 00:00:00", "2014-01-03 00:00:00");

        createAppointment(getDate("2014-01-02"), getDate("2014-01-03"), schedule);
        Iterator<Slot> query3 = createIterator("2014-01-01", "2014-01-03", schedule);
        assertFalse(query3.hasNext());
    }

    /**
     * Verifies that slots are returned when a schedule starts at 0:00.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotFor0HourStart(boolean maxDuration) {
        Entity schedule = createSchedule("00:00", "17:00", maxDuration);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query, schedule, "2014-01-01 00:00:00", "2014-01-01 17:00:00");
        checkSlot(query, schedule, "2014-01-02 00:00:00", "2014-01-02 17:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Verifies that slots are returned when a schedule ends at 24:00.
     *
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     */
    private void checkFindFreeSlotFor24HourEnd(boolean maxDuration) {
        Entity schedule = createSchedule("09:00", "24:00", maxDuration);
        Iterator<Slot> query = createIterator("2014-01-01", "2014-01-03", schedule);
        checkSlot(query, schedule, "2014-01-01 09:00:00", "2014-01-02 00:00:00");
        checkSlot(query, schedule, "2014-01-02 09:00:00", "2014-01-03 00:00:00");
        assertFalse(query.hasNext());
    }

    /**
     * Creates a new query.
     *
     * @param fromDate  the query from date
     * @param toDate    the query to date
     * @param schedules the schedules to query
     * @return a new query
     */
    private FreeSlotQuery createQuery(String fromDate, String toDate, Entity... schedules) {
        FreeSlotQuery query = new FreeSlotQuery(getArchetypeService(), appointmentService, rosterService,
                                                appointmentRules);
        query.setFromDate(getDate(fromDate));
        query.setToDate(getDate(toDate));
        query.setSchedules(schedules);
        return query;
    }

    /**
     * Creates a new query iterator.
     *
     * @param fromDate  the query from date
     * @param toDate    the query to date
     * @param schedules the schedules to query
     * @return a new query
     */
    private Iterator<Slot> createIterator(String fromDate, String toDate, Entity... schedules) {
        FreeSlotQuery query = createQuery(fromDate, toDate, schedules);
        return createIterator(query);
    }

    /**
     * Creates a new query iterator.
     *
     * @param query the query
     * @return the query iterator
     */
    private Iterator<Slot> createIterator(FreeSlotQuery query) {
        long start = System.currentTimeMillis();
        Iterator<Slot> iterator = query.query();
        long end = System.currentTimeMillis();
        System.out.println("Executed query in " + (end - start) + "ms");
        return iterator;
    }

    /**
     * Verifies that the next slot returned by the iterator matches that expected.
     *
     * @param iterator  the slot iterator
     * @param schedule  the expected schedule
     * @param startTime the expected slot start time
     * @param endTime   the expected slot end time
     */
    private void checkSlot(Iterator<Slot> iterator, Entity schedule, String startTime, String endTime) {
        checkSlot(iterator, schedule, getDatetime(startTime), getDatetime(endTime));
    }

    /**
     * Verifies that the next slot returned by the iterator matches that expected.
     *
     * @param iterator  the slot iterator
     * @param schedule  the expected schedule
     * @param startTime the expected slot start time
     * @param endTime   the expected slot end time
     */
    private void checkSlot(Iterator<Slot> iterator, Entity schedule, Date startTime, Date endTime) {
        assertTrue(iterator.hasNext());
        Slot slot = iterator.next();
        assertEquals(schedule.getId(), slot.getSchedule());
        checkDate(startTime, slot.getStartTime());
        checkDate(endTime, slot.getEndTime());
    }

    /**
     * Verifies that a date matches that expected,
     *
     * @param expected the expected date
     * @param actual   the actual date
     */
    private void checkDate(Date expected, Date actual) {
        assertEquals("expected=" + expected + ", actual=" + actual, 0, DateRules.compareTo(expected, actual));
    }

    private Period getTime(String time) {
        PeriodFormatterBuilder builder = new PeriodFormatterBuilder().appendHours().appendLiteral(":").appendMinutes();
        PeriodFormatter formatter = builder.toFormatter();
        return formatter.parsePeriod(time);
    }

    /**
     * Creates a schedule.
     *
     * @param startTime   the schedule start time. May be {@code null}
     * @param endTime     the schedule end time. May be {@code null}
     * @param maxDuration if {@code true}, add a 24-hour maxDuration to the schedule, to limit index range scans
     * @return the schedule
     */
    private Entity createSchedule(String startTime, String endTime, boolean maxDuration) {
        TestScheduleBuilder builder = schedulingFactory.newSchedule();
        builder.location(location).times(startTime, endTime);
        if (maxDuration) {
            builder.maxDuration(24, DateUnits.HOURS);
        } else {
            builder.noMaxDuration();
        }
        return builder.build();
    }

    /**
     * Creates a schedule.
     *
     * @param startTime the schedule start time. May be {@code null}
     * @param endTime   the schedule end time. May be {@code null}
     * @param cageType  the cage type
     * @return the schedule
     */
    private Entity createSchedule(String startTime, String endTime, Entity cageType) {
        return schedulingFactory.newSchedule().location(location).times(startTime, endTime).cageType(cageType).build();
    }

    /**
     * Creates an appointment.
     *
     * @param startTime the start time
     * @param endTime   the end time
     * @param schedule  the schedule
     */
    private void createAppointment(Date startTime, Date endTime, Entity schedule) {
        newAppointment(schedule).startTime(startTime).endTime(endTime).build();
    }

    private TestAppointmentBuilder newAppointment(Entity schedule) {
        return schedulingFactory.newAppointment().schedule(schedule).customer(customerFactory.createCustomer()).patient(patientFactory.createPatient()).appointmentType(schedulingFactory.createAppointmentType());
    }

    /**
     * Creates an appointment.
     *
     * @param startTime the start time
     * @param endTime   the end time
     * @param schedule  the schedule
     */
    private void createAppointment(String startTime, String endTime, Entity schedule) {
        newAppointment(schedule).startTime(startTime).endTime(endTime).build();
    }

    /**
     * Creates an appointment.
     *
     * @param startTime the start time
     * @param endTime   the end time
     * @param schedule  the schedule
     * @param clinician the clinician
     */
    private void createAppointment(String startTime, String endTime, Entity schedule, User clinician) {
        newAppointment(schedule).startTime(startTime).endTime(endTime).clinician(clinician).build();
    }
}
