/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2024 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.product.io;

import au.com.bytecode.opencsv.CSVWriter;
import org.junit.Before;
import org.junit.Test;
import org.openvpms.archetype.function.date.DateFunctions;
import org.openvpms.archetype.rules.doc.DocumentHandler;
import org.openvpms.archetype.rules.doc.DocumentHandlers;
import org.openvpms.archetype.rules.finance.tax.TaxRules;
import org.openvpms.archetype.rules.product.PricingGroup;
import org.openvpms.archetype.rules.product.ProductPriceRules;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.test.builder.lookup.TestLookupFactory;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.component.business.domain.im.document.Document;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.model.product.ProductPrice;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;

import static java.math.BigDecimal.ZERO;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.openvpms.archetype.rules.math.MathRules.ONE_HUNDRED;

/**
 * Tests the {@link ProductCSVWriter} and {@link ProductCSVReader} classes.
 *
 * @author Tim Anderson
 */
public class ProductCSVWriterReaderTestCase extends AbstractProductIOTest {

    /**
     * The lookup factory.
     */
    @Autowired
    private TestLookupFactory lookupFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * The product price rules.
     */
    private ProductPriceRules rules;

    /**
     * The tax rules.
     */
    private TaxRules taxRules;

    /**
     * The document handlers.
     */
    private DocumentHandlers handlers;

    /**
     * The first fixed price, pricing group A.
     */
    private ProductPrice fixed1A;

    /**
     * The first fixed price, pricing group B.
     */
    private ProductPrice fixed1B;

    /**
     * The first fixed price, no pricing group.
     */
    private ProductPrice fixed1C;

    /**
     * The second fixed price, pricing group A.
     */
    private ProductPrice fixed2A;

    /**
     * The second fixed price, pricing group B.
     */
    private ProductPrice fixed2B;

    /**
     * The second fixed price, no pricing group.
     */
    private ProductPrice fixed2C;

    /**
     * The first unit price, pricing group A.
     */
    private ProductPrice unit1A;

    /**
     * The first unit price, pricing group B.
     */
    private ProductPrice unit1B;

    /**
     * The first unit price, no pricing group.
     */
    private ProductPrice unit1C;

    /**
     * The second unit price.
     */
    private ProductPrice unit2A;

    /**
     * The second unit price.
     */
    private ProductPrice unit2B;

    /**
     * The second unit price.
     */
    private ProductPrice unit2C;

    /**
     * The product.
     */
    private Product product;

    /**
     * Pricing group A.
     */
    private Lookup groupA;

    /**
     * Pricing group B.
     */
    private Lookup groupB;

    /**
     * Current date - 2 months and 1 day.
     */
    private Date monthsMinus21;

    /**
     * Sets up the test case.
     */
    @Before
    public void setUp() {
        rules = new ProductPriceRules(getArchetypeService());
        Party practice = practiceFactory.getPractice();
        taxRules = new TaxRules(practice, getArchetypeService());
        handlers = new DocumentHandlers(getArchetypeService());

        Date today = DateRules.getToday();
        Date tomorrow = DateRules.getTomorrow();
        Date yesterday = DateRules.getYesterday();
        Date months2 = DateRules.getDate(today, 2, DateUnits.MONTHS);
        Date months21 = DateRules.getDate(months2, 1, DateUnits.DAYS);
        Date monthsMinus2 = DateRules.getDate(today, -2, DateUnits.MONTHS);
        monthsMinus21 = DateRules.getDate(monthsMinus2, -1, DateUnits.DAYS);

        groupA = productFactory.createPricingGroup("A");
        groupB = productFactory.createPricingGroup("B");

        fixed1A = newFixedPrice("1.0", "0.5", "100", "10", monthsMinus21, yesterday, false)
                .pricingGroups(groupA)
                .build();
        fixed1B = newFixedPrice("1.0", "0.5", "100", "10", monthsMinus21, yesterday, false)
                .pricingGroups(groupB)
                .build();
        fixed1C = createFixedPrice("1.0", "0.5", "100", "10", monthsMinus21, yesterday, false);
        fixed2A = newFixedPrice("1.08", "0.6", "80", "10", today, months2, true)
                .pricingGroups(groupA)
                .build();
        fixed2B = newFixedPrice("1.08", "0.6", "80", "10", today, months2, true)
                .pricingGroups(groupB)
                .build();
        fixed2C = createFixedPrice("1.08", "0.6", "80", "10", today, months2, true);

        unit1A = newUnitPrice("1.92", "1.2", "60", "10", monthsMinus2, today)
                .pricingGroups(groupA)
                .build();
        unit1B = newUnitPrice("1.92", "1.2", "60", "10", monthsMinus2, today)
                .pricingGroups(groupB)
                .build();
        unit1C = createUnitPrice("1.92", "1.2", "60", "10", monthsMinus2, today);

        unit2A = newUnitPrice("2.55", "1.5", "70", "10", tomorrow, months21)
                .pricingGroups(groupA)
                .build();
        unit2B = newUnitPrice("2.55", "1.5", "70", "10", tomorrow, months21)
                .pricingGroups(groupB)
                .build();
        unit2C = createUnitPrice("2.55", "1.5", "70", "10", tomorrow, months21);

        product = newProduct("Product A", "A")
                .addTaxTypes(lookupFactory.createTaxType(5))
                .addPrices(fixed1A, fixed1B, fixed1C, fixed2A, fixed2B, fixed2C,
                           unit1A, unit1B, unit1C, unit2A, unit2B, unit2C)
                .build();
    }

    /**
     * Tests writing the latest prices, and reading them back again.
     */
    @Test
    public void testWriteReadLatestPrices() {
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), true, true,
                                         new PricingGroup(groupA, false)); // exclude prices with no group

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());

        List<SimpleDateFormat> dateFormats = reader.getDateFormats(document);
        assertEquals(1, dateFormats.size());
        assertEquals("yy-MM-dd", dateFormats.get(0).toPattern());
        reader.setDateFormats(dateFormats);

        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(1, data.getFixedPrices().size());
        checkPrice(data.getFixedPrices().get(0), fixed2A);

        assertEquals(1, data.getUnitPrices().size());
        checkPrice(data.getUnitPrices().get(0), unit2A);
    }

    /**
     * Tests writing all prices for a pricing group, and reading them back again.
     */
    @Test
    public void testWriteReadAllPricesForGroup() {
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true, new
                PricingGroup(groupA, false));

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(2, data.getFixedPrices().size());
        checkPrice(data.getFixedPrices().get(0), fixed2A);
        checkPrice(data.getFixedPrices().get(1), fixed1A);

        assertEquals(2, data.getUnitPrices().size());
        checkPrice(data.getUnitPrices().get(0), unit2A);
        checkPrice(data.getUnitPrices().get(1), unit1A);
    }

    /**
     * Tests writing all prices, and reading them back again.
     */
    @Test
    public void testWriteReadAllPrices() {
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true, PricingGroup.ALL);

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        checkPrices(data.getFixedPrices(), fixed1A, fixed1B, fixed1C, fixed2A, fixed2B, fixed2C);
        checkPrices(data.getUnitPrices(), unit1A, unit1B, unit1C, unit2A, unit2B, unit2C);
    }

    /**
     * Tests writing prices matching a date range, and reading them back again.
     */
    @Test
    public void testWriteReadRangePrices() {
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Date from = monthsMinus21;
        Date to = DateRules.getDate(from, 1, DateUnits.MONTHS);
        Document document = writer.write(Collections.singletonList(product).iterator(), from, to, true,
                                         new PricingGroup(groupB, false));

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(1, data.getFixedPrices().size());
        checkPrice(data.getFixedPrices().get(0), fixed1B);

        assertEquals(1, data.getUnitPrices().size());
        checkPrice(data.getUnitPrices().get(0), unit1B);
    }

    /**
     * Tests writing a product that contains just unit prices.
     */
    @Test
    public void testWriteUnitPrices() {
        product.removeProductPrice(fixed1A);
        product.removeProductPrice(fixed1B);
        product.removeProductPrice(fixed1C);
        product.removeProductPrice(fixed2A);
        product.removeProductPrice(fixed2B);
        product.removeProductPrice(fixed2C);
        save(product);
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true,
                                         new PricingGroup(groupA, false));

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(0, data.getFixedPrices().size());
        assertEquals(2, data.getUnitPrices().size());
        checkPrice(data.getUnitPrices().get(0), unit2A);
        checkPrice(data.getUnitPrices().get(1), unit1A);
    }

    /**
     * Tests writing a product that contains just fixed  prices.
     */
    @Test
    public void testWriteFixedPrices() {
        product.removeProductPrice(unit1A);
        product.removeProductPrice(unit1B);
        product.removeProductPrice(unit1C);
        product.removeProductPrice(unit2A);
        product.removeProductPrice(unit2B);
        product.removeProductPrice(unit2C);
        save(product);
        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true,
                                         new PricingGroup(groupA, false));

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(2, data.getFixedPrices().size());
        assertEquals(0, data.getUnitPrices().size());
        checkPrice(data.getFixedPrices().get(0), fixed2A);
        checkPrice(data.getFixedPrices().get(1), fixed1A);
    }

    /**
     * Verifies that the correct date formats are detected in the input document.
     */
    @Test
    public void testDateParsing() {
        ProductPrice fixed1 = createFixedPrice("1.0", "0.5", "100", "10", "2012-02-01", "2012-04-01", false);
        ProductPrice fixed2 = createFixedPrice("1.08", "0.6", "80", "10", "2012-04-02", "2012-06-01", true);
        ProductPrice unit1 = createUnitPrice("1.92", "1.2", "60", "10", "2012-02-02", "2012-04-02");
        ProductPrice unit2 = createUnitPrice("2.55", "1.5", "70", "10", "2012-04-03", "2012-06-02");
        Product product = newProduct("Product A", "A")
                .addPrices(fixed1, fixed2, unit1, unit2)
                .addTaxTypes(lookupFactory.createTaxType(5))
                .build();

        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers) {
            @Override
            protected String getDate(Date date) {
                return new DateFunctions().format(date, "dd/MM/yy");
            }
        };
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true, PricingGroup.ALL);
        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        List<SimpleDateFormat> dateFormats = reader.getDateFormats(document);
        assertEquals(3, dateFormats.size());
        assertEquals(ProductCSVReader.getDayMonthYearFormats().get(0), dateFormats.get(0));
        assertEquals(ProductCSVReader.getYearMonthDayFormats().get(0), dateFormats.get(1));
        assertEquals(ProductCSVReader.getMonthDayYearFormats().get(0), dateFormats.get(2));
        reader.setDateFormats(Collections.singletonList(dateFormats.get(0)));
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        checkPrices(data.getFixedPrices(), fixed1, fixed2);
        checkPrices(data.getUnitPrices(), unit1, unit2);
    }

    /**
     * Verifies that an error is raised if a fixed price is specified without a fixed cost.
     *
     * @throws IOException for any I/O error
     */
    @Test
    public void testMissingFixedCost() throws IOException {
        String[][] data = {{"1001", "Product A", "A", "-1", "1.08", "", "10", "02/04/12", "01/06/12", "true", "", "-1",
                            "2.55", "1.5", "10", "03/04/12", "02/06/12", "", "5.0"}};
        ProductDataSet products = createProductDataSet(data);
        assertEquals(1, products.getErrors().size());
        assertEquals("A value for Fixed Cost is required", products.getErrors().get(0).getError());
    }

    /**
     * Verifies that an error is raised if a fixed price is specified without a fixed max discount.
     *
     * @throws IOException for any I/O error
     */
    @Test
    public void testMissingFixedMaxDiscount() throws IOException {
        String[][] data = {{"1001", "Product A", "A", "-1", "1.08", "0.6", "", "02/04/12", "01/06/12", "true", "", "-1",
                            "2.55", "1.5", "10", "03/04/12", "02/06/12", "", "5.0"}};
        ProductDataSet products = createProductDataSet(data);

        assertEquals(1, products.getErrors().size());
        assertEquals("A value for Fixed Price Max Discount is required", products.getErrors().get(0).getError());
    }

    /**
     * Verifies that an error is raised if a unit price is specified without a unit cost.
     *
     * @throws IOException for any I/O error
     */
    @Test
    public void testMissingUnitCost() throws IOException {
        String[][] data = {{"1001", "Product A", "A", "-1", "1.08", "0.6", "10", "02/04/12", "01/06/12", "true", "",
                            "-1", "2.55", "", "10", "03/04/12", "02/06/12", "", "5.0"}};
        ProductDataSet products = createProductDataSet(data);

        assertEquals(1, products.getErrors().size());
        assertEquals("A value for Unit Cost is required", products.getErrors().get(0).getError());
    }

    /**
     * Verifies that an error is raised if a unit price is specified without a unit price max discount.
     *
     * @throws IOException for any I/O error
     */
    @Test
    public void testMissingUnitPriceMaxDiscount() throws IOException {
        String[][] data = {{"1001", "Product A", "A", "-1", "1.08", "0.6", "10", "02/04/12", "01/06/12", "true", "",
                            "-1", "2.55", "1.5", "", "03/04/12", "02/06/12", "", "5.0"}};
        ProductDataSet products = createProductDataSet(data);

        assertEquals(1, products.getErrors().size());
        assertEquals("A value for Unit Price Max Discount is required", products.getErrors().get(0).getError());
    }

    /**
     * Verifies that the line no. is reported if a line is invalid.
     *
     * @throws IOException for any I/O error
     */
    @Test
    public void testInvalidLine() throws IOException {
        String[][] data = {{"1001", "Product A", "A", "-1", "1.08", "0.6", "10", "02/04/12", "01/06/12", "true", "",
                            "-1", "2.55", "1.5", "10", "03/04/12", "02/06/12", "", "5.0", ""},
                           {"1002", "Product B", "B", "-1", "1.08", "0.6", "10", "02/04/12", "01/06/12", "true", "",
                            "-1", "2.55", "1.5", "10", "03/04/12", "02/06/12", "", "5.0"},
                           // one short - this is OK as Notes is optional
                           {"1003", "Product C", "C", "-1", "1.08", "0.6", "10", "02/04/12", "01/06/12", "true", "",
                            "-1", "2.55", "1.5", "10", "03/04/12", "02/06/12", ""}}; // two columns short
        ProductDataSet products = createProductDataSet(data);
        assertEquals("Line 4 contains 18 fields, but 20 are required", products.getErrors().get(0).getError());
    }

    /**
     * Verifies that products can be written if they have prices with {@code null} costs and maxDiscounts.
     * <p/>
     * These will default to 0.0 and 100 respectively, as per the price archetypes.
     */
    @Test
    public void testWritePricesWithNullCostAndMaxDiscounts() {
        ProductPrice fixed1 = createFixedPrice("1.0", null, "100", null, "2012-02-01", "2012-04-01", false);
        ProductPrice unit1 = createUnitPrice("1.92", null, "60", null, "2012-02-02", "2012-04-02");
        Product product = newProduct("Product A", "A", fixed1, unit1)
                .addClassifications(lookupFactory.createTaxType(5))
                .build();

        ProductCSVWriter writer = new ProductCSVWriter(getArchetypeService(), rules, taxRules, handlers);
        Document document = writer.write(Collections.singletonList(product).iterator(), false, true, PricingGroup.ALL);

        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        reader.setDateFormats(ProductCSVReader.getYearMonthDayFormats());
        ProductDataSet products = reader.read(document);
        assertEquals(1, products.getData().size());
        assertEquals(0, products.getErrors().size());

        ProductData data = products.getData().get(0);
        checkProduct(data, product);
        assertEquals(1, data.getFixedPrices().size());
        checkEquals(ZERO, data.getFixedPrices().get(0).getCost());
        checkEquals(ONE_HUNDRED, data.getFixedPrices().get(0).getMaxDiscount());

        assertEquals(1, data.getUnitPrices().size());
        checkEquals(ZERO, data.getUnitPrices().get(0).getCost());
        checkEquals(ONE_HUNDRED, data.getUnitPrices().get(0).getMaxDiscount());
    }

    /**
     * Creates a CSV containing the supplied data, and reads it back into a {@link ProductDataSet}.
     *
     * @param data the data to write
     * @return the read data
     * @throws IOException for any I/O error
     */
    private ProductDataSet createProductDataSet(String[][] data) throws IOException {
        StringWriter writer = new StringWriter();
        CSVWriter csv = new CSVWriter(writer, ProductCSVWriter.SEPARATOR);
        csv.writeNext(ProductCSVWriter.HEADER);
        for (String[] line : data) {
            csv.writeNext(line);
        }
        csv.close();

        DocumentHandler handler = handlers.get("Dummy.csv", ProductCSVReader.MIME_TYPE);
        Document document = handler.create("Dummy.csv", new ByteArrayInputStream(
                                                   writer.toString().getBytes(StandardCharsets.UTF_8)),
                                           ProductCSVReader.MIME_TYPE, -1);
        ProductCSVReader reader = new ProductCSVReader(handlers, getLookupService());
        return reader.read(document);
    }

    /**
     * Verifies a product matches that expected.
     *
     * @param data     the product data
     * @param expected the expected product
     */
    private void checkProduct(ProductData data, Product expected) {
        IMObjectBean bean = getBean(expected);
        assertEquals(expected.getId(), data.getId());
        assertEquals(expected.getName(), data.getName());
        assertEquals(bean.getString("printedName"), data.getPrintedName());
        checkEquals(taxRules.getTaxRate(expected), data.getTaxRate());
    }

    /**
     * Verifies a price matches that expected.
     *
     * @param data     the price data
     * @param expected the expected price
     */
    private void checkPrice(PriceData data, ProductPrice expected) {
        IMObjectBean bean = getBean(expected);
        assertEquals(expected.getPrice(), data.getPrice());
        assertEquals(bean.getBigDecimal("cost"), data.getCost());
        assertEquals(bean.getBigDecimal("maxDiscount"), data.getMaxDiscount());
        assertEquals(expected.getFromDate(), data.getFrom());
        assertEquals(expected.getToDate(), data.getTo());
        Set<Lookup> pricingGroups = ProductIOHelper.getPricingGroups(expected, getArchetypeService());
        assertEquals(pricingGroups, data.getPricingGroups());
    }

    private void checkPrices(List<PriceData> actual, ProductPrice... expected) {
        assertEquals(actual.size(), expected.length);
        for (ProductPrice price : expected) {
            for (PriceData other : actual) {
                if (other.getId() == price.getId()) {
                    checkPrice(other, price);
                    return;
                }
            }
            fail("PriceData not found for id=" + price.getId());
        }
    }

}
