/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.patient;

import org.junit.Before;
import org.junit.Test;
import org.openvpms.archetype.rules.practice.PracticeRules;
import org.openvpms.archetype.rules.practice.PracticeService;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.customer.TestCustomerFactory;
import org.openvpms.archetype.test.builder.patient.TestPatientFactory;
import org.openvpms.archetype.test.builder.supplier.TestSupplierFactory;
import org.openvpms.component.math.WeightUnits;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.entity.EntityIdentity;
import org.openvpms.component.model.entity.EntityRelationship;
import org.openvpms.component.model.object.PeriodRelationship;
import org.openvpms.component.model.object.Relationship;
import org.openvpms.component.model.party.Party;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Date;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.openvpms.archetype.test.TestHelper.getDate;


/**
 * Tests the {@link PatientRules} class.
 *
 * @author Tim Anderson
 */
public class PatientRulesTestCase extends ArchetypeServiceTest {

    /**
     * The practice service.
     */
    @Autowired
    private PracticeService practiceService;

    /**
     * The customer factory.
     */
    @Autowired
    private TestCustomerFactory customerFactory;

    /**
     * The patient factory.
     */
    @Autowired
    private TestPatientFactory patientFactory;

    /**
     * The supplier factory.
     */
    @Autowired
    private TestSupplierFactory supplierFactory;

    /**
     * The rules.
     */
    private PatientRules rules;

    /**
     * The practice rules.
     */
    private PracticeRules practiceRules;


    /**
     * Sets up the test case.
     */
    @Before
    public void setUp() {
        practiceRules = new PracticeRules(getArchetypeService(), null);
        rules = new PatientRules(practiceRules, practiceService, getArchetypeService(), getLookupService());
    }

    /**
     * Tests the {@link PatientRules#getOwner} and {@link PatientRules#getOwnerReference} methods.
     */
    @Test
    public void testGetOwner() {
        Party patient1 = patientFactory.createPatient();
        assertNull(rules.getOwner(patient1));
        assertNull(rules.getOwnerReference(patient1));

        Party customer = customerFactory.createCustomer();
        Party patient2 = patientFactory.createPatient();
        Party customer2 = customerFactory.createCustomer();
        Party customer3 = customerFactory.createCustomer();

        rules.addPatientLocationRelationship(customer2, patient2);
        rules.addPatientOwnerRelationship(customer3, patient2);
        deactivateRelationship(patient2, PatientArchetypes.PATIENT_OWNER);

        rules.addPatientOwnerRelationship(customer, patient2);

        assertEquals(customer, rules.getOwner(patient2));
        assertEquals(customer.getObjectReference(), rules.getOwnerReference(patient2));

        deactivateRelationship(patient2, PatientArchetypes.PATIENT_OWNER);
        assertNull(rules.getOwner(patient2));
        assertNull(rules.getOwnerReference(patient2));
    }

    /**
     * Tests the {@link PatientRules#getOwner(Act)} method.
     */
    @Test
    public void testGetOwnerFromAct() {
        Party patient = patientFactory.createPatient();
        Party customer1 = customerFactory.createCustomer();
        Party customer2 = customerFactory.createCustomer();

        checkOwner(patient, "2006-12-10", null); // no owner

        EntityRelationship r1 = rules.addPatientOwnerRelationship(customer1, patient);
        EntityRelationship r2 = rules.addPatientOwnerRelationship(customer2, patient);

        r1.setActiveStartTime(getDate("2007-01-01"));
        r1.setActiveEndTime(getDate("2007-01-31"));
        r2.setActiveStartTime(getDate("2007-02-02"));
        r2.setActiveEndTime(null);

        save(patient);
        save(customer1);
        save(customer2);

        checkOwner(patient, "2006-12-10", customer1); // customer1 closest
        checkOwner(patient, "2007-01-01", customer1); // exact match
        checkOwner(patient, "2007-01-31", customer2); // customer1 relationship ended
        checkOwner(patient, "2007-02-01", customer2); // customer2 closest
        checkOwner(patient, "2007-02-02", customer2); // exact match
        checkOwner(patient, "2008-01-01", customer2); // unbounded end time
    }

    /**
     * Tests the {@link PatientRules#isOwner} method.
     */
    @Test
    public void testIsOwner() {
        Party patient1 = patientFactory.createPatient();
        Party customer1 = customerFactory.createCustomer();
        Party patient2 = patientFactory.createPatient();
        Party customer2 = customerFactory.createCustomer();
        rules.addPatientOwnerRelationship(customer1, patient2);
        rules.addPatientLocationRelationship(customer2, patient2);

        assertFalse(rules.isOwner(customer1, patient1));
        assertTrue(rules.isOwner(customer1, patient2));

        deactivateRelationship(patient2, PatientArchetypes.PATIENT_OWNER);
        assertFalse(rules.isOwner(customer1, patient2));

    }

    /**
     * Test the {@link PatientRules#getLocation} methods
     */
    @Test
    public void testGetLocation() {
        Party patient1 = patientFactory.createPatient();
        assertNull(rules.getLocation(patient1));

        Party customer = customerFactory.createCustomer();
        Party patient2 = patientFactory.createPatient();
        rules.addPatientLocationRelationship(customer, patient2);
        assertEquals(customer, rules.getLocation(patient2));

        deactivateRelationship(patient2, PatientArchetypes.PATIENT_LOCATION);
        assertNull(rules.getOwner(patient2));
        assertNull(rules.getOwnerReference(patient2));
    }

    /**
     * Tests the {@link PatientRules#getLocation(Act)} method.
     */
    @Test
    public void testGetLocationFromAct() {
        Party patient = patientFactory.createPatient();
        Party customer1 = customerFactory.createCustomer();
        Party customer2 = customerFactory.createCustomer();

        checkLocation(patient, "2006-12-10", null); // no owner

        EntityRelationship r1 = rules.addPatientLocationRelationship(customer1, patient);
        EntityRelationship r2 = rules.addPatientLocationRelationship(customer2, patient);

        r1.setActiveStartTime(getDate("2007-01-01"));
        r1.setActiveEndTime(getDate("2007-01-31"));
        r2.setActiveStartTime(getDate("2007-02-02"));
        r2.setActiveEndTime(null);

        save(patient);
        save(customer1);
        save(customer2);

        checkLocation(patient, "2006-12-10", customer1); // customer1 closest
        checkLocation(patient, "2007-01-01", customer1); // exact match
        checkLocation(patient, "2007-01-31", customer2); // customer1 relationship ended
        checkLocation(patient, "2007-02-01", customer2); // customer2 closest
        checkLocation(patient, "2007-02-02", customer2); // exact match
        checkLocation(patient, "2008-01-01", customer2); // unbounded end time
    }

    /**
     * Tests the {@link PatientRules#getReferralVet} method.
     */
    @Test
    public void testGetReferralVet() {
        Party vet = supplierFactory.createVet();
        Party patient = patientFactory.newPatient()
                .addReferredFrom(vet, new Date())
                .build(false);

        IMObjectBean bean = getBean(patient);
        PeriodRelationship referral = bean.getObject("referrals", PeriodRelationship.class);
        assertNotNull(referral);
        assertTrue(referral.isA(PatientArchetypes.REFERRED_FROM));

        // verify the vet is returned for a time > the default start time
        Party vet2 = rules.getReferralVet(patient, new Date());
        assertEquals(vet, vet2);

        // now set the start and end time and verify that there is no referrer
        // for a later time (use time addition due to system clock granularity)
        Date start = new Date();
        Date end = new Date(start.getTime() + 1);
        Date later = new Date(end.getTime() + 1);
        referral.setActiveStartTime(start);
        referral.setActiveEndTime(end);
        assertNull(rules.getReferralVet(patient, later));
    }

    /**
     * Tests the {@link PatientRules#getPatientWeight(Party)} method.
     */
    @Test
    public void testGetPatientWeight() {
        Party patient = patientFactory.createPatient();
        assertNull(rules.getPatientWeight(patient));

        patientFactory.newWeight()
                .patient(patient)
                .startTime("2006-12-22")
                .weight(5)
                .build();
        assertEquals("5 Kilograms", rules.getPatientWeight(patient));

        patientFactory.newWeight()
                .patient(patient)
                .startTime("2007-02-25")
                .weight(13, WeightUnits.POUNDS)
                .build();
        assertEquals("13 Pounds", rules.getPatientWeight(patient));
    }

    /**
     * Tests the {@link PatientRules#setInactive(Party)} method.
     */
    @Test
    public void testSetInactive() {
        Party patient = patientFactory.newPatient().build(false);
        assertTrue(patient.isActive());

        rules.setInactive(patient);
        patient = get(patient);
        assertFalse(patient.isActive());
    }

    /**
     * Tests the {@link PatientRules#setDeceased(Party)} and
     * {@link PatientRules#isDeceased(Party)} methods.
     */
    @Test
    public void testDeceased() {
        Party patient = patientFactory.newPatient().build(false);
        assertFalse(rules.isDeceased(patient));
        rules.setDeceased(patient);
        patient = get(patient);
        assertTrue(rules.isDeceased(patient));
    }

    /**
     * Tests the {@link PatientRules#setDesexed(Party)} and
     * {@link PatientRules#isDesexed(Party)} methods.
     */
    @Test
    public void testDesexed() {
        Party patient = patientFactory.newPatient().build(false);
        assertFalse(rules.isDesexed(patient));
        rules.setDesexed(patient);
        patient = get(patient);
        assertTrue(rules.isDesexed(patient));
    }

    /**
     * Tests the {@link PatientRules#getMicrochipNumber(Party)} method.
     */
    @Test
    public void testGetMicrochipNumber() {
        Party patient = patientFactory.newPatient().build(false);
        assertNull(rules.getMicrochipNumber(patient));

        EntityIdentity microchip = patientFactory.createMicrochip("1234567");
        patient.addIdentity(microchip);
        assertEquals("1234567", rules.getMicrochipNumber(patient));

        microchip.setActive(false);
        assertNull(rules.getMicrochipNumber(patient));
    }

    /**
     * Tests the {@link PatientRules#getMicrochipNumbers(Party)} method.
     */
    @Test
    public void testGetMicrochipNumbers() {
        Party patient = patientFactory.newPatient().build(false);
        assertNull(rules.getMicrochipNumbers(patient));

        EntityIdentity microchip1 = patientFactory.createMicrochip("123");
        patient.addIdentity(microchip1);
        save(patient);
        assertEquals("123", rules.getMicrochipNumbers(patient));

        EntityIdentity microchip2 = patientFactory.createMicrochip("456");
        patient.addIdentity(microchip2);
        save(patient); // 456 will be returned first as its id is higher

        assertEquals("456, 123", rules.getMicrochipNumbers(patient));
    }

    /**
     * Tests the {@link PatientRules#getPetTag(Party)} method.
     */
    @Test
    public void testGetPetTag() {
        Party patient = patientFactory.newPatient().build(false);
        assertNull(rules.getPetTag(patient));

        EntityIdentity tag = create(PatientArchetypes.PET_TAG, EntityIdentity.class);
        IMObjectBean tagBean = getBean(tag);
        tagBean.setValue("petTag", "1234567");
        patient.addIdentity(tag);
        assertEquals("1234567", rules.getPetTag(patient));

        tag.setActive(false);
        assertNull(rules.getPetTag(patient));
    }

    /**
     * Tests the {@link PatientRules#getCage(Party)} method.
     */
    @Test
    public void testGetCage() {
        Party patient = patientFactory.newPatient().build(false);
        assertNull(rules.getCage(patient));

        patientFactory.updatePatient(patient)
                .cage("PURPLE_1")
                .build(false);
        assertEquals("Purple 1", rules.getCage(patient));
    }

    /**
     * Tests the {@link PatientRules#getRabiesTag(Party)} method.
     */
    @Test
    public void testGetRabiesTag() {
        Party patient = patientFactory.newPatient().build(false);
        assertNull(rules.getRabiesTag(patient));

        EntityIdentity tag = create(PatientArchetypes.RABIES_TAG, EntityIdentity.class);
        tag.setIdentity("1234567");
        patient.addIdentity(tag);
        assertEquals("1234567", rules.getRabiesTag(patient));

        tag.setActive(false);
        assertNull(rules.getRabiesTag(patient));
    }

    /**
     * Tests {@link PatientRules#getPatientAge} for a patient with no birth date.
     */
    @Test
    public void testGetPatientAgeNoBirthDate() {
        Party patient = patientFactory.newPatient().build(false);
        String age = rules.getPatientAge(patient);
        assertEquals("No Birthdate", age);
    }

    /**
     * Tests {@link PatientRules#getPatientAge} for a patient with a birth date.
     */
    @Test
    public void testGetPatientAgeWithBirthDate() {
        Date birthDate = getDate("2010-01-01");
        Party patient = patientFactory.newPatient()
                .dateOfBirth("2010-01-01")
                .build(false);
        String age = rules.getPatientAge(patient);
        PatientAgeFormatter formatter = new PatientAgeFormatter(getLookupService(), practiceRules,
                                                                getArchetypeService());
        String expected = formatter.format(birthDate);
        assertEquals(expected, age);
    }

    /**
     * Tests {@link PatientRules#getPatientAge} for a deceased patient.
     */
    @Test
    public void testGetPatientAgeWithDeceasedDate() {
        Date birth = getDate("2010-01-01");
        Date deceased = getDate("2011-05-01");
        Party patient = patientFactory.newPatient()
                .dateOfBirth(birth)
                .deceased(true)
                .dateOfDeath(deceased)
                .build(false);
        IMObjectBean bean = getBean(patient);
        bean.setValue("dateOfBirth", birth);
        bean.setValue("deceasedDate", deceased);
        String age = rules.getPatientAge(patient);
        PatientAgeFormatter formatter = new PatientAgeFormatter(getLookupService(), practiceRules,
                                                                getArchetypeService());
        String expected = formatter.format(birth, deceased);
        assertEquals(expected, age);
    }

    /**
     * Tests the {@link PatientRules#isAllergy(Act)} method.
     */
    @Test
    public void testIsAllergy() {
        Party patient = patientFactory.createPatient();
        Entity other = patientFactory.createAlertType("Z Alert Type 1");
        Entity allergy = patientFactory.createAlertType("Z Alert Type 2", "ALLERGY");
        Entity aggression = patientFactory.createAlertType("Z Alert Type 3", "AGGRESSION");
        Act act1 = patientFactory.createAlert(patient, other);
        assertFalse(rules.isAllergy(act1));

        Act act2 = patientFactory.createAlert(patient, allergy);
        assertTrue(rules.isAllergy(act2));

        Act act3 = patientFactory.createAlert(patient, aggression);
        assertFalse(rules.isAllergy(act3));
    }

    /**
     * Checks the ownership of a patient for a given date.
     *
     * @param patient  the patient
     * @param date     the date
     * @param expected the expected owner
     */
    private void checkOwner(Party patient, String date, Party expected) {
        Act act = patientFactory.newVisit()
                .patient(patient)
                .startTime(date)
                .build(false);
        assertEquals(expected, rules.getOwner(act));
    }

    /**
     * Checks the location customer of a patient for a given date.
     *
     * @param patient  the patient
     * @param date     the date
     * @param expected the expected location customer
     */
    private void checkLocation(Party patient, String date, Party expected) {
        Act act = patientFactory.newVisit()
                .patient(patient)
                .startTime(date)
                .build(false);
        assertEquals(expected, rules.getLocation(act));
    }

    /**
     * Marks a relationship inactive.
     *
     * @param patient   the patient
     * @param shortName the relationship short name
     */
    private void deactivateRelationship(Party patient, String shortName) {
        for (Relationship relationship : patient.getEntityRelationships()) {
            if (relationship.isA(shortName)) {
                relationship.setActive(false);
            }
        }
        try {
            Thread.sleep(1000);
        } catch (InterruptedException ignore) {
            // do nothing
        }
    }
}
