/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.statement;

import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.finance.account.AbstractCustomerAccountTest;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.test.TestHelper;
import org.openvpms.component.business.service.archetype.helper.TypeHelper;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.party.Party;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CLOSING_BALANCE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.OPENING_BALANCE;


/**
 * Base class for statement test cases.
 *
 * @author Tim Anderson
 */
public class AbstractStatementTest extends AbstractCustomerAccountTest {

    /**
     * The practice.
     */
    private Party practice;

    /**
     * Returns the practice.
     *
     * @return the practice
     */
    protected Party getPractice() {
        if (practice == null) {
            practice = create("party.organisationPractice", Party.class);
            practice.setName("Z Test Practice");
        }
        return practice;
    }

    /**
     * Helper to create a date-time given a string of the form
     * <em>yyyy-mm-dd hh:mm:ss</em>.
     *
     * @param value the value
     * @return the corresponding date-time
     */
    protected Date getDatetime(String value) {
        return TestHelper.getDatetime(value);
    }

    /**
     * Helper to create a date given a string of the form <em>yyyy-mm-dd</em>.
     *
     * @param value the value
     * @return the corresponding date
     */
    protected Date getDate(String value) {
        return TestHelper.getDate(value);
    }

    /**
     * Returns all acts for a statement date.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the acts for the statement date
     */
    protected List<Act> getActs(Party customer, Date statementDate) {
        StatementActHelper helper
                = new StatementActHelper(getArchetypeService());
        Date timestamp = helper.getStatementTimestamp(statementDate);
        Iterable<Act> acts = helper.getActs(customer, timestamp);
        List<Act> result = new ArrayList<>();
        for (Act act : acts) {
            result.add(act);
        }
        return result;
    }

    /**
     * Returns all posted acts for a statement date.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the posted acts for the statement date
     */
    protected List<Act> getPostedActs(Party customer, Date statementDate) {
        StatementActHelper helper
                = new StatementActHelper(getArchetypeService());
        Date open = helper.getOpeningBalanceTimestamp(customer, statementDate);
        Date close = helper.getClosingBalanceTimestamp(customer, statementDate,
                                                       open);
        Iterable<FinancialAct> acts = helper.getPostedActs(customer, open, close, true);
        List<Act> result = new ArrayList<>();
        for (Act act : acts) {
            result.add(act);
        }
        return result;
    }

    /**
     * Verifies a debit adjustment matches that expected.
     *
     * @param act               the act to verify
     * @param amount            the expected amount
     * @param accountFeeMessage the account fee message
     */
    protected void checkDebitAdjust(Act act, BigDecimal amount, String accountFeeMessage) {
        checkAct(act, CustomerAccountArchetypes.DEBIT_ADJUST, amount);
        IMObjectBean bean = getBean(act);
        assertEquals(accountFeeMessage, bean.getString("notes"));
    }

    /**
     * Verifies that an act matches the expected short name and amount, and
     * is POSTED.
     *
     * @param act       the act to verify
     * @param shortName the expected archetype short name
     * @param amount    the expected amount
     */
    protected void checkAct(Act act, String shortName, BigDecimal amount) {
        checkAct(act, shortName, amount, FinancialActStatus.POSTED);
    }

    /**
     * Verifies that an act matches the expected short name and amount, and
     * status.
     *
     * @param act       the act to verify
     * @param shortName the expected archetype short name
     * @param amount    the expected amount
     * @param status    the expected act status
     */
    protected void checkAct(Act act, String shortName, BigDecimal amount,
                            String status) {
        assertTrue(TypeHelper.isA(act, shortName));
        assertEquals(status, act.getStatus());
        checkEquals(amount, ((FinancialAct) act).getTotal());
    }

    /**
     * Verfies an act matches the expected act and status.
     *
     * @param act      the act to verify
     * @param expected the expected act
     * @param status   the expected status
     */
    protected void checkAct(Act act, FinancialAct expected, String status) {
        assertEquals(expected, act);
        checkAct(act, expected.getArchetype(), expected.getTotal(), status);
    }

    /**
     * Verifies an act appears in a list of acts with the expected status.
     *
     * @param acts     the acts
     * @param expected the expected act
     * @param status   the expected status
     */
    protected void checkAct(List<Act> acts, FinancialAct expected,
                            String status) {
        for (Act act : acts) {
            if (act.equals(expected)) {
                checkAct(act, expected, status);
                return;
            }
        }
        fail("Expected act " + expected + " not found in acts");
    }

    /**
     * Verifies that an opening balance matches that expected.
     *
     * @param act    the act to verify
     * @param amount the expected amount
     */
    protected void checkOpeningBalance(Act act, BigDecimal amount) {
        checkAct(act, OPENING_BALANCE, amount, FinancialActStatus.POSTED);
    }

    /**
     * Verifies that a closing balance matches that expected.
     *
     * @param act     the act to verify
     * @param amount  the expected amount
     * @param overdue the expected overdue amount
     */
    protected void checkClosingBalance(Act act, BigDecimal amount, BigDecimal overdue) {
        checkAct(act, CLOSING_BALANCE, amount, FinancialActStatus.POSTED);
        IMObjectBean bean = getBean(act);
        checkEquals(overdue, bean.getBigDecimal("overdueBalance"));
    }
}
