/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.customer.TestCustomerFactory;
import org.openvpms.archetype.test.builder.customer.account.AbstractTestSingleAccountActBuilder;
import org.openvpms.archetype.test.builder.customer.account.TestCustomerAccountFactory;
import org.openvpms.archetype.test.builder.patient.TestPatientFactory;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.archetype.test.builder.product.TestProductFactory;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;


/**
 * Base class for customer account test cases.
 *
 * @author Tim Anderson
 */
public abstract class AbstractCustomerAccountTest extends ArchetypeServiceTest {

    /**
     * The customer account rules.
     */
    @Autowired
    protected CustomerAccountRules accountRules;

    /**
     * The customer account factory.
     */
    @Autowired
    protected TestCustomerAccountFactory accountFactory;

    /**
     * The customer factory.
     */
    @Autowired
    protected TestCustomerFactory customerFactory;

    /**
     * The patient factory.
     */
    @Autowired
    protected TestPatientFactory patientFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * The product factory.
     */
    @Autowired
    protected TestProductFactory productFactory;

    /**
     * The customer.
     */
    private Party customer;

    /**
     * The patient.
     */
    private Party patient;

    /**
     * The product.
     */
    private Product product;

    /**
     * The stock location.
     */
    private Party stockLocation;

    /**
     * The till.
     */
    private Entity till;


    /**
     * Returns the customer.
     *
     * @return the customer
     */
    protected Party getCustomer() {
        if (customer == null) {
            customer = customerFactory.createCustomer();
        }
        return customer;
    }

    /**
     * Sets the customer.
     *
     * @param customer the customer
     */
    protected void setCustomer(Party customer) {
        this.customer = customer;
    }

    /**
     * Returns the patient.
     *
     * @return the patient
     */
    protected Party getPatient() {
        if (patient == null) {
            patient = patientFactory.createPatient(getCustomer());
        }
        return patient;
    }

    /**
     * Returns the product.
     *
     * @return the product
     */
    protected Product getProduct() {
        if (product == null) {
            product = productFactory.createMedication();
        }
        return product;
    }

    /**
     * Returns the customer account rules.
     *
     * @return the rules
     */
    protected CustomerAccountRules getRules() {
        return accountRules;
    }

    /**
     * Returns the till.
     *
     * @return the till
     */
    protected Entity getTill() {
        if (till == null) {
            till = practiceFactory.createTill();
        }
        return till;
    }

    /**
     * Returns the stock location.
     *
     * @return the stock location
     */
    protected Party getStockLocation() {
        if (stockLocation == null) {
            stockLocation = practiceFactory.createStockLocation();
        }
        return stockLocation;
    }

    /**
     * Helper to create a <em>POSTED</em> <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount the act total
     * @return a list containing the invoice act and its item
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount) {
        return createChargesInvoice(amount, getCustomer());
    }

    /**
     * Helper to create a <em>POSTED</em>  <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount   the act total
     * @param customer the customer
     * @return a list containing the invoice act and its item
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Party customer) {
        return createChargesInvoice(amount, customer, getProduct());
    }

    /**
     * Helper to create a <em>POSTED</em>  <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount   the act total
     * @param customer the customer
     * @param product  the product
     * @return a list containing the invoice act and its item
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Party customer, Product product) {
        List<FinancialAct> result = FinancialTestHelper.createChargesInvoice(
                amount, customer, getPatient(), product, ActStatus.POSTED);
        IMObjectBean item = getBean(result.get(1));
        item.setTarget("stockLocation", getStockLocation());
        return result;
    }

    /**
     * Helper to create a <em>POSTED</em>
     * <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount    the act total
     * @param customer  the customer
     * @param startTime the act start time
     * @return a list containing the invoice act and its item
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Party customer, Date startTime) {
        List<FinancialAct> acts = createChargesInvoice(amount, customer);
        acts.get(0).setActivityStartTime(startTime);
        return acts;
    }

    /**
     * Helper to create a <em>POSTED</em>
     * <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount    the act total
     * @param startTime the act start time
     * @return a list containing the invoice act and its item
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Date startTime) {
        List<FinancialAct> acts = createChargesInvoice(amount);
        acts.get(0).setActivityStartTime(startTime);
        return acts;
    }

    /**
     * Helper to create an <em>act.customerAccountChargesInvoice</em> with an
     * associated <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount    the act total
     * @param customer  the customer
     * @param startTime the act start time
     * @param status    the act status
     * @return a new act
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Party customer, Date startTime,
                                                      String status) {
        List<FinancialAct> acts = FinancialTestHelper.createChargesInvoice(
                amount, customer, getPatient(), getProduct(), status);
        acts.get(0).setActivityStartTime(startTime);
        return acts;
    }

    /**
     * Helper to create a <em>POSTED</em>
     * <em>act.customerAccountChargesInvoice</em> with an associated
     * <em>act.customerAccountInvoiceItem</em>.
     *
     * @param amount    the act total
     * @param startTime the act start time
     * @param status    the act status
     * @return a new act
     */
    protected List<FinancialAct> createChargesInvoice(BigDecimal amount, Date startTime, String status) {
        return createChargesInvoice(amount, getCustomer(), startTime, status);
    }

    /**
     * Helper to create a <em>POSTED</em>
     * <em>act.customerAccountChargesCounter</em> with an associated
     * <em>act.customerAccountCreditItem</em>.
     *
     * @param amount the act total
     * @return a new act
     */
    protected List<FinancialAct> createChargesCounter(BigDecimal amount) {
        List<FinancialAct> result = FinancialTestHelper.createChargesCounter(amount, getCustomer(), getProduct(),
                                                                             ActStatus.POSTED);
        IMObjectBean item = getBean(result.get(1));
        item.setTarget("stockLocation", getStockLocation());
        return result;
    }

    /**
     * Helper to create an <em>POSTED</em>
     * <em>act.customerAccountChargesCredit</em> associated with a
     * <em>customerAccountCreditItem</em>.
     *
     * @param amount the act total
     * @return a new act
     */
    protected List<FinancialAct> createChargesCredit(BigDecimal amount) {
        List<FinancialAct> result = FinancialTestHelper.createChargesCredit(amount, getCustomer(), getPatient(),
                                                                            getProduct(), ActStatus.POSTED);
        IMObjectBean item = getBean(result.get(1));
        item.setTarget("stockLocation", getStockLocation());
        return result;
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em>.
     *
     * @param amount the act total
     * @return a new payment
     */
    protected FinancialAct createPayment(int amount) {
        return createPayment(BigDecimal.valueOf(amount), getCustomer());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em>.
     *
     * @param amount the act total
     * @return a new payment
     */
    protected FinancialAct createPayment(BigDecimal amount) {
        return createPayment(amount, getCustomer());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em>.
     *
     * @param amount   the act total
     * @param customer the customer
     * @return a new payment
     */
    protected FinancialAct createPayment(BigDecimal amount, Party customer) {
        return FinancialTestHelper.createPaymentCash(amount, customer, getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em>.
     *
     * @param amount    the act total
     * @param startTime the act start time
     * @return a new act
     */
    protected FinancialAct createPayment(BigDecimal amount, Date startTime) {
        FinancialAct payment = createPayment(amount);
        payment.setActivityStartTime(startTime);
        return payment;
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentCash</em>.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentCash(BigDecimal amount) {
        return FinancialTestHelper.createPaymentCash(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentCheque</em>.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentCheque(BigDecimal amount) {
        return FinancialTestHelper.createPaymentCheque(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentCredit</em>.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentCredit(BigDecimal amount) {
        return FinancialTestHelper.createPaymentCredit(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentDiscount</em>.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentDiscount(BigDecimal amount) {
        return FinancialTestHelper.createPaymentDiscount(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentEFT</em>.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentEFT(BigDecimal amount) {
        return FinancialTestHelper.createPaymentEFT(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create a POSTED <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentOther</em>.
     * <p/>
     * The item is saved, but the payment isn't, to avoid updating the balance.
     *
     * @param amount the amount
     * @return a new payment
     */
    protected FinancialAct createPaymentOther(BigDecimal amount) {
        return createPaymentOther(amount, null);
    }

    /**
     * Helper to create a POSTED <em>act.customerAccountPayment</em> containing an
     * <em>act.customerAccountPaymentOther</em>.
     * <p/>
     * The item is saved, but the payment isn't, to avoid updating the balance.
     *
     * @param amount      the amount
     * @param paymentType the custom payment tyep
     * @return a new payment
     */
    protected FinancialAct createPaymentOther(BigDecimal amount, String paymentType) {
        FinancialAct item = accountFactory.newOtherPaymentItem()
                .amount(amount)
                .paymentType(paymentType)
                .build();
        return accountFactory.newPayment()
                .customer(getCustomer())
                .till(getTill())
                .add(item)
                .status(FinancialActStatus.POSTED)
                .build(false);
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefund(BigDecimal amount) {
        return FinancialTestHelper.createRefundCash(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em> containing an <em>act.customerAccountRefundCash</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundCash(BigDecimal amount) {
        return FinancialTestHelper.createRefundCash(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em> containing an <em>act.customerAccountRefundCheque</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundCheque(BigDecimal amount) {
        return FinancialTestHelper.createRefundCheque(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em> containing an <em>act.customerAccountRefundCredit</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundCredit(BigDecimal amount) {
        return FinancialTestHelper.createRefundCredit(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em> containing an <em>act.customerAccountRefundDiscount</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundDiscount(BigDecimal amount) {
        return FinancialTestHelper.createRefundDiscount(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create an <em>act.customerAccountRefund</em> containing an <em>act.customerAccountRefundEFT</em>.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundEFT(BigDecimal amount) {
        return FinancialTestHelper.createRefundEFT(amount, getCustomer(), getTill());
    }

    /**
     * Helper to create a POSTED <em>act.customerAccountRefund</em> containing an
     * <em>act.customerAccountRefundOther</em>.
     * <p/>
     * The item is saved, but the refund isn't, to avoid updating the balance.
     *
     * @param amount the act total
     * @return a new refund
     */
    protected FinancialAct createRefundOther(BigDecimal amount) {
        return createRefundOther(amount, null);
    }

    /**
     * Helper to create a POSTED <em>act.customerAccountRefund</em> containing an
     * <em>act.customerAccountRefundOther</em>.
     * <p/>
     * The item is saved, but the refund isn't, to avoid updating the balance.
     *
     * @param amount      the act total
     * @param paymentType the custom payment type
     * @return a new refund
     */
    public FinancialAct createRefundOther(BigDecimal amount, String paymentType) {
        FinancialAct item = accountFactory.newOtherRefundItem()
                .amount(amount)
                .paymentType(paymentType)
                .build();
        return accountFactory.newRefund()
                .customer(getCustomer())
                .till(getTill())
                .add(item)
                .status(FinancialActStatus.POSTED)
                .build(false);
    }

    /**
     * Helper to create an <em>act.customerAccountCreditAdjust</em>.
     *
     * @param amount the act total
     * @return a new credit adjustment
     */
    protected FinancialAct createCreditAdjust(BigDecimal amount) {
        return createSingleAccountAct(accountFactory.newCreditAdjust(), amount);
    }

    /**
     * Helper to create an <em>act.customerAccountDebitAdjust</em>.
     *
     * @param amount the act total
     * @return a new debit adjustment
     */
    protected FinancialAct createDebitAdjust(BigDecimal amount) {
        return createSingleAccountAct(accountFactory.newDebitAdjust(), amount);
    }

    /**
     * Helper to create an <em>act.customerAccountInitialBalance</em>.
     *
     * @param amount the act total
     * @return a new initial balance
     */
    protected FinancialAct createInitialBalance(BigDecimal amount) {
        return createSingleAccountAct(accountFactory.newInitialBalance(), amount);
    }

    /**
     * Helper to create an <em>act.customerAccountBadDebt</em>.
     *
     * @param amount the act total
     * @return a new bad debt
     */
    protected FinancialAct createBadDebt(BigDecimal amount) {
        return createSingleAccountAct(accountFactory.newBadDebt(), amount);
    }

    /**
     * Uses a {@link AbstractTestSingleAccountActBuilder} to build a POSTED unsaved act.
     *
     * @param builder the builder
     * @param amount  the amount
     * @return the act
     */
    protected FinancialAct createSingleAccountAct(AbstractTestSingleAccountActBuilder<?> builder, BigDecimal amount) {
        return builder.amount(amount)
                .customer(getCustomer())
                .status(FinancialActStatus.POSTED)
                .build(false);
    }

    /**
     * Helper to create and save a new <em>lookup.customerAccountType</em>
     * classification.
     *
     * @param paymentTerms the payment terms
     * @param paymentUom   the payment units
     * @return a new classification
     */
    protected Lookup createAccountType(int paymentTerms, DateUnits paymentUom) {
        return createAccountType(paymentTerms, paymentUom, BigDecimal.ZERO);
    }

    /**
     * Helper to create and save a new <em>lookup.customerAccountType</em>
     * classification.
     *
     * @param paymentTerms     the payment terms
     * @param paymentUom       the payment units
     * @param accountFeeAmount the account fee
     * @return a new classification
     */
    protected Lookup createAccountType(int paymentTerms, DateUnits paymentUom, BigDecimal accountFeeAmount) {
        return customerFactory.newAccountType().paymentTerms(paymentTerms, paymentUom)
                .accountFee(accountFeeAmount, AccountType.FeeType.FIXED)
                .build();
    }
}
