/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.openvpms.archetype.rules.finance.eft.EFTPOSTransactionStatus;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.model.bean.Predicates;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.util.List;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;

/**
 * Rules for EFT and payment processor transactions linked to customer account act items.
 *
 * @author Tim Anderson
 */
public class TransactionRules {

    public static class ItemStatus {

        public enum Status {
            NO_TRANSACTION, // No transaction present
            UNSUBMITTED,    // Transaction needs to be submitted
            INCOMPLETE,     // Transaction submitted, but not completed
            COMPLETE,       // Transaction is complete
        }

        private final Status status;

        private final FinancialAct item;

        private final FinancialAct transaction;

        public ItemStatus(Status status) {
            this(status, null, null);
        }

        public ItemStatus(Status status, FinancialAct item, FinancialAct transaction) {
            this.status = status;
            this.item = item;
            this.transaction = transaction;
        }

        public boolean isUnsubmitted() {
            return status == Status.UNSUBMITTED;
        }

        public boolean isIncomplete() {
            return status == Status.INCOMPLETE;
        }

        public boolean isComplete() {
            return status == Status.COMPLETE;
        }

        public Status getStatus() {
            return status;
        }

        public FinancialAct getItem() {
            return item;
        }

        public FinancialAct getTransaction() {
            return transaction;
        }
    }

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * Items relationship node name.
     */
    private static final String ITEMS = "items";

    /**
     * Constructs a {@link TransactionRules}.
     *
     * @param service the archetype service
     */
    public TransactionRules(ArchetypeService service) {
        this.service = service;
    }

    /**
     * Determines if a payment or refund has an EFTPOS transaction with APPROVED status.
     *
     * @param act the payment/refund
     * @return {@code true} if at least one transaction is APPROVED, otherwise {@code false}
     */
    public boolean hasApprovedEFTPOSTransaction(FinancialAct act) {
        boolean result = false;
        List<FinancialAct> items = getEFTItems(act);
        for (FinancialAct item : items) {
            if (itemHasApprovedEFTPOSTransaction(item)) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Determines if an EFT item has an approved EFTPOS transaction.
     *
     * @param item the EFT item
     * @return {@code true} if the item has approved EFTPOS transaction, otherwise {@code false}
     */
    public boolean itemHasApprovedEFTPOSTransaction(FinancialAct item) {
        boolean result = false;
        if (item.isA(PAYMENT_EFT, REFUND_EFT)) {
            FinancialAct latest = getMostRecentTransaction(item);
            result = (latest != null && EFTPOSTransactionStatus.APPROVED.equals(latest.getStatus()));
        }
        return result;
    }

    /**
     * Returns the most recent transaction from an item (i.e. an EFT or payment processor payment/refund).
     *
     * @param item the item
     * @return the most recent transaction, or {@code null} if there are no transactions
     */
    public FinancialAct getMostRecentTransaction(FinancialAct item) {
        List<FinancialAct> transactions = service.getBean(item)
                .getTargets("transactions", FinancialAct.class, Policies.orderBySequence(false));
        return !transactions.isEmpty() ? transactions.get(0) : null;
    }

    /**
     * Returns the {@link CustomerAccountArchetypes#PAYMENT_EFT} or {@link CustomerAccountArchetypes#REFUND_EFT}
     * items associated with a payment or refund.
     *
     * @param act the payment/refund
     * @return the corresponding EFT items
     */
    public List<FinancialAct> getEFTItems(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.getTargets(ITEMS, FinancialAct.class,
                               Policies.all(Predicates.targetIsA(PAYMENT_EFT, REFUND_EFT)));
    }

    /**
     * Returns the {@link CustomerAccountArchetypes#PAYMENT_PP} or {@link CustomerAccountArchetypes#REFUND_PP}
     * items associated with a payment or refund.
     *
     * @param act the payment/refund
     * @return the corresponding payment processor items
     */
    public List<FinancialAct> getPaymentProcessorItems(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.getTargets(ITEMS, FinancialAct.class, Policies.all(Predicates.targetIsA(PAYMENT_PP, REFUND_PP)));
    }

    /**
     * Determines if a payment or refund has an outstanding EFT transaction.
     * <p/>
     * This is true for any:
     * <ul>
     *     <li><em>act.customerAccountPaymentEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSPayment</em></li>
     *     <li><em>act.customerAccountRefundEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSRefund</em></li>
     * </ul>
     * NOTE that this doesn't apply to payments and refunds where there is no integrated EFTPOS support; at least
     * one <em>act.EFTPOSPayment</em> or <em>act.EFTPOSRefund</em> must be present.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is DECLINED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding EFT transaction.
     */
    public ItemStatus getEFTPOSTransactionStatus(FinancialAct act, boolean ignoreUnsuccessful) {
        ItemStatus result = new ItemStatus(ItemStatus.Status.NO_TRANSACTION);
        for (FinancialAct item : getEFTItems(act)) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest == null) {
                // offline EFT items don't have a transaction
                break;
            } else {
                String status = latest.getStatus();
                if (EFTPOSTransactionStatus.PENDING.equals(status)
                    || EFTPOSTransactionStatus.IN_PROGRESS.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                } else if (EFTPOSTransactionStatus.APPROVED.equals(status)
                           || EFTPOSTransactionStatus.NO_TERMINAL.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.COMPLETE, item, latest);
                } else if (!ignoreUnsuccessful) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                }
            }
        }
        return result;
    }

    /**
     * Determines if a payment or refund has an outstanding payment processor item.
     * <p/>
     * This is any payment processor item without a COMPLETED transaction.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is CANCELLED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding payment processor transaction,
     * otherwise {@code false}
     */
    public ItemStatus getPaymentProcessorTransactionStatus(FinancialAct act, boolean ignoreUnsuccessful) {
        ItemStatus result = new ItemStatus(ItemStatus.Status.NO_TRANSACTION);
        for (FinancialAct item : getPaymentProcessorItems(act)) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest == null) {
                result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, null);
                break;
            } else {
                String status = latest.getStatus();
                if (PaymentProcessorTransactionStatus.PENDING.equals(status)
                    || PaymentProcessorTransactionStatus.IN_PROGRESS.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, latest);
                    break;
                } else if (PaymentProcessorTransactionStatus.SUBMITTED.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                    break;
                } else if (PaymentProcessorTransactionStatus.COMPLETED.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.COMPLETE);
                } else if (!ignoreUnsuccessful) {
                    result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, latest);
                    break;
                }
            }
        }
        return result;
    }
}