/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.openvpms.component.model.act.FinancialAct;

/**
 * Determines if a customer account act may be posted.
 *
 * @author Tim Anderson
 */
public class PostStatus {

    /**
     * The internal status.
     */
    private final Status status;

    /**
     * The line item. May be {@code null}
     */
    private final FinancialAct item;

    /**
     * The EFT/payment processor transaction. May be {@code null}
     */
    private final FinancialAct transaction;

    /**
     * Constructs a {@link PostStatus}.
     *
     * @param status the status
     */
    private PostStatus(Status status) {
        this(status, null, null);
    }

    /**
     * Constructs a {@link PostStatus}.
     *
     * @param status      the status
     * @param item        the line item. May be {@code null}
     * @param transaction the EFT/payment processor transaction. May be {@code null}
     */
    private PostStatus(Status status, FinancialAct item, FinancialAct transaction) {
        this.status = status;
        this.item = item;
        this.transaction = transaction;
    }

    /**
     * Determines if the act can be posted.
     *
     * @return {@code true} if the act can be posted, otherwise {@code false}
     */
    public boolean canPost() {
        return status == Status.CAN_POST;
    }

    /**
     * Determines if the act has an outstanding EFTPOS transaction, prevening it being posted.
     *
     * @return {@code true} if the act has an outstanding EFTPOS transaction, otherwise {@code false}
     */
    public boolean hasOutstandingEFTPOSTransaction() {
        return status == Status.OUTSTANDING_EFTPOS_TRANSACTION;
    }

    /**
     * Determines if the act has an unsubmitted payment processor transaction, prevening it being posted.
     *
     * @return {@code true} if the act has an unsubmitted transaction, otherwise {@code false}
     */
    public boolean hasUnsubmittedPaymentProcessorTransaction() {
        return status == Status.UNSUBMITTED_PAYMENT_PROCESSOR_TRANSACTION;
    }

    /**
     * Determines if the act has an outstanding payment processor transaction, prevening it being posted.
     *
     * @return {@code true} if the act has an outstanding payment processor transaction, otherwise {@code false}
     */
    public boolean hasOutstandingPaymentProcessorTransaction() {
        return status == Status.OUTSTANDING_PAYMENT_PROCESSOR_TRANSACTION;
    }

    /**
     * Returns the line item preventing the act from being posted.
     *
     * @return the line item. May be {@code null}
     */
    public FinancialAct getItem() {
        return item;
    }

    /**
     * Returns the EFT/payment processor transaction preventing the act from being posted.
     *
     * @return the transaction. May be {@code null}
     */
    public FinancialAct getTransaction() {
        return transaction;
    }

    /**
     * Constructs {@link PostStatus}, indicating that the act can be posted.
     *
     * @return the status
     */
    public static PostStatus postingAllowed() {
        return new PostStatus(Status.CAN_POST);
    }

    /**
     * Constructs {@link PostStatus}, indicating posting is unsupported.
     *
     * @return the status
     */
    public static PostStatus unsupported() {
        return new PostStatus(Status.UNSUPPORTED);
    }

    /**
     * Constructs {@link PostStatus}, indicating the act is already posted.
     *
     * @return the status
     */
    public static PostStatus posted() {
        return new PostStatus(Status.POSTED);
    }

    /**
     * Constructs {@link PostStatus}, indicating the act has an outstanding EFTPOS transaction.
     *
     * @param item        the EFTPOS line item
     * @param transaction the EFTPOS transaction
     * @return the status
     */
    public static PostStatus outstandingEFTPOSTransaction(FinancialAct item, FinancialAct transaction) {
        return new PostStatus(Status.OUTSTANDING_EFTPOS_TRANSACTION, item, transaction);
    }

    /**
     * Constructs {@link PostStatus}, indicating the act has an unsubmitted payment processor transaction.
     *
     * @param item        the payment processor line item
     * @param transaction the payment processor transaction
     * @return the status
     */
    public static PostStatus unsubmittedPaymentProcessorTransaction(FinancialAct item, FinancialAct transaction) {
        return new PostStatus(Status.UNSUBMITTED_PAYMENT_PROCESSOR_TRANSACTION, item, transaction);
    }

    /**
     * Constructs {@link PostStatus}, indicating the act has an outstanding payment processor transaction.
     *
     * @param item        the payment processor line item
     * @param transaction the payment processor transaction
     * @return the status
     */
    public static PostStatus outstandingPaymentProcessorTransaction(FinancialAct item, FinancialAct transaction) {
        return new PostStatus(Status.OUTSTANDING_PAYMENT_PROCESSOR_TRANSACTION, item, transaction);
    }

    private enum Status {

        UNSUPPORTED, POSTED, CAN_POST, OUTSTANDING_EFTPOS_TRANSACTION, UNSUBMITTED_PAYMENT_PROCESSOR_TRANSACTION,
        OUTSTANDING_PAYMENT_PROCESSOR_TRANSACTION
    }
}
