/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.openvpms.archetype.rules.act.ActCopyHandler;
import org.openvpms.archetype.rules.finance.eft.EFTPOSTransactionStatus;
import org.openvpms.component.business.domain.im.archetype.descriptor.ArchetypeDescriptor;
import org.openvpms.component.business.domain.im.archetype.descriptor.NodeDescriptor;
import org.openvpms.component.business.service.archetype.helper.IMObjectCopier;
import org.openvpms.component.business.service.archetype.helper.IMObjectCopyHandler;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.util.Arrays;
import java.util.List;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.BAD_DEBT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.COUNTER;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.COUNTER_ITEM;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.COUNTER_ITEM_RELATIONSHIP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CREDIT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CREDIT_ADJUST;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CREDIT_ITEM;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CREDIT_ITEM_RELATIONSHIP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.DEBIT_ADJUST;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.INITIAL_BALANCE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.INVOICE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.INVOICE_ITEM;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.INVOICE_ITEM_RELATIONSHIP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_CASH;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_CHEQUE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_CREDIT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_DISCOUNT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_ITEM_RELATIONSHIP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_OTHER;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_CASH;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_CHEQUE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_CREDIT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_DISCOUNT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_ITEM_RELATIONSHIP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_OTHER;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;


/**
 * {@link IMObjectCopyHandler} that creates reversals for customer acts.
 *
 * @author Tim Anderson
 */
class CustomerActReversalHandler extends ActCopyHandler {

    /**
     * Determines if the reversal will be POSTED. If so payment processor and APPROVED EFT items cannot be reversed
     * as these need to be handled via a 3rd-party service.
     */
    private final boolean post;

    /**
     * Map of debit types to their corresponding credit types.
     */
    static final String[][] TYPE_MAP = {
            {INVOICE, CREDIT},
            {INVOICE_ITEM, CREDIT_ITEM},
            {INVOICE_ITEM_RELATIONSHIP, CREDIT_ITEM_RELATIONSHIP},
            {COUNTER, CREDIT},
            {COUNTER_ITEM, CREDIT_ITEM},
            {COUNTER_ITEM_RELATIONSHIP, CREDIT_ITEM_RELATIONSHIP},
            {PAYMENT, REFUND},
            {PAYMENT_ITEM_RELATIONSHIP, REFUND_ITEM_RELATIONSHIP},
            {PAYMENT_CASH, REFUND_CASH},
            {PAYMENT_CHEQUE, REFUND_CHEQUE},
            {PAYMENT_CREDIT, REFUND_CREDIT},
            {PAYMENT_DISCOUNT, REFUND_DISCOUNT},
            {PAYMENT_EFT, REFUND_EFT},
            {PAYMENT_OTHER, REFUND_OTHER},
            {PAYMENT_PP, REFUND_PP},
            {DEBIT_ADJUST, CREDIT_ADJUST},
            {DEBIT_ADJUST, BAD_DEBT},
            {INITIAL_BALANCE, CREDIT_ADJUST}
    };

    /**
     * Names of nodes that can't be copied.
     */
    private final List<String> UNCOPYABLE = Arrays.asList("credit", "allocatedAmount", "accountBalance", "allocation",
                                                          "reversals", "reverses", "estimates", "hide", "tillBalance",
                                                          "audit", "transaction", "transactions", "reversal");

    /**
     * Constructs a {@link CustomerActReversalHandler}.
     *
     * @param act                         the act to reverse
     * @param post determines if the reversal will be POSTED. If so payment processor and APPROVED EFT items cannot be
     *             reversed as these need to be handled via a 3rd-party service.
     */
    CustomerActReversalHandler(Act act, boolean post) {
        super(TYPE_MAP);
        setReverse(act.isA(CREDIT, REFUND, CREDIT_ADJUST, BAD_DEBT));
        this.post = post;
    }

    /**
     * Determines how {@link IMObjectCopier} should treat an object.
     *
     * @param object  the source object
     * @param service the archetype service
     * @return {@code object} if the object shouldn't be copied, {@code null} if it should be replaced with
     * {@code null}, or a new instance if the object should be copied
     */
    @Override
    public IMObject getObject(IMObject object, ArchetypeService service) {
        IMObject result = super.getObject(object, service);
        if (object.isA(REFUND_CASH) && result.isA(PAYMENT_CASH)) {
            IMObjectBean refund = service.getBean(object);
            IMObjectBean payment = service.getBean(result);
            payment.setValue("tendered", refund.getBigDecimal("roundedAmount"));
        } else if (object.isA(PAYMENT_EFT)) {
            FinancialAct latest = getMostRecentTransaction((FinancialAct) object, service);
            if (latest != null && EFTPOSTransactionStatus.APPROVED.equals(latest.getStatus())) {
                if (post) {
                    // EFT refunds need to be submitted to an EFTPOS service before posting
                    throw new IllegalStateException("Cannot reverse " + object.getArchetype());
                }
                addReversal(object, result, REFUND_EFT, service);
            }
        } else if (object.isA(REFUND_EFT)) {
            FinancialAct latest = getMostRecentTransaction((FinancialAct) object, service);
            if (latest != null && EFTPOSTransactionStatus.APPROVED.equals(latest.getStatus())) {
                // cannot reverse EFT refunds linked to a service
                throw new IllegalStateException("Cannot reverse " + object.getArchetype());
            }
        } else if (object.isA(PAYMENT_PP)) {
            if (post) {
                // payment processor refunds need to be submitted to a payment processor
                // service before posting
                throw new IllegalStateException("Cannot reverse " + object.getArchetype());
            }
            addReversal(object, result, REFUND_PP, service);
        } else if (object.isA(REFUND_PP)) {
            // cannot reverse payment processor refunds
            throw new IllegalStateException("Cannot reverse " + object.getArchetype());
        }
        return result;
    }

    /**
     * Returns the most recent transaction from an item (i.e. an EFT or payment processor payment/refund).
     *
     * @param item the item
     * @return the most recent transaction, or {@code null} if there are no transactions
     */
    private FinancialAct getMostRecentTransaction(FinancialAct item, ArchetypeService service) {
        TransactionRules rules = new TransactionRules(service);
        return rules.getMostRecentTransaction(item);
    }

    /**
     * Adds a reversal relationship between two acts.
     *
     * @param object    the object being reversed
     * @param reversal  the reversal
     * @param archetype the expected archetype of the revesral
     * @param service   the archetype service
     */
    private void addReversal(IMObject object, IMObject reversal, String archetype, ArchetypeService service) {
        if (!reversal.isA(archetype)) {
            throw new IllegalStateException("Invalid reversal " + reversal.getArchetype());
        } else {
            IMObjectBean payment = service.getBean(object);
            payment.addTarget("reversal", reversal, "reverses");
        }
    }

    /**
     * Helper to determine if a node is copyable.
     *
     * @param archetype the archetype descriptor
     * @param node      the node descriptor
     * @param source    if {@code true} the node is the source; otherwise it's the target
     * @return {@code true} if the node is copyable; otherwise {@code false}
     */
    @Override
    protected boolean isCopyable(ArchetypeDescriptor archetype, NodeDescriptor node, boolean source) {
        String name = node.getName();
        if (UNCOPYABLE.contains(name)) {
            return false;
        } else {
            return super.isCopyable(archetype, node, source);
        }
    }
}