/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.supplier;

import org.openvpms.archetype.test.builder.act.AbstractTestActVerifier;
import org.openvpms.archetype.test.builder.object.ValueStrategy;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.product.Product;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;

import static org.junit.Assert.fail;

/**
 * Verifies <em>act.supplierOrderItem</em>, <em>act.supplierDeliveryItem</em> and <em>act.supplierReturnItem</em>
 * instances match that expected, for testing purposes.
 *
 * @author Tim Anderson
 */
public class TestSupplierActItemVerifier<B extends TestSupplierActItemVerifier<B>>
        extends AbstractTestActVerifier<FinancialAct, B> {

    /**
     * The expected product.
     */
    private ValueStrategy product = ValueStrategy.value(null);

    /**
     * The expected reorder code.
     */
    private ValueStrategy reorderCode = ValueStrategy.value(null);

    /**
     * The expected reorder description.
     */
    private ValueStrategy reorderDescription = ValueStrategy.value(null);

    /**
     * The expected package size.
     */
    private ValueStrategy packageSize = ValueStrategy.value(null);

    /**
     * The expected package units.
     */
    private ValueStrategy packageUnits = ValueStrategy.value(null);

    /**
     * The expected quantity.
     */
    private ValueStrategy quantity = ValueStrategy.value(null);

    /**
     * The expected unit price.
     */
    private ValueStrategy unitPrice = ValueStrategy.value(null);

    /**
     * The expected list price.
     */
    private ValueStrategy listPrice = ValueStrategy.value(null);

    /**
     * The expected tax.
     */
    private ValueStrategy tax = ValueStrategy.value(null);

    /**
     * The expected total.
     */
    private ValueStrategy total = ValueStrategy.value(null);

    /**
     * Constructs a {@link TestSupplierActItemVerifier}.
     *
     * @param service the archetype service
     */
    public TestSupplierActItemVerifier(ArchetypeService service) {
        super(service);
    }

    /**
     * Initialises this from an item.
     *
     * @param item the item
     * @return this
     */
    public B initialise(FinancialAct item) {
        IMObjectBean bean = getBean(item);
        initialise(bean);
        return getThis();
    }

    /**
     * Sets the expected product.
     *
     * @param product the product
     * @return this
     */
    public B product(Product product) {
        return product((product != null) ? product.getObjectReference() : null);
    }

    /**
     * Sets the expected product.
     *
     * @param product the product
     * @return this
     */
    public B product(Reference product) {
        this.product = ValueStrategy.value(product);
        return getThis();
    }

    /**
     * Sets the expected reorder code.
     *
     * @param reorderCode the reorder code
     * @return this
     */
    public B reorderCode(String reorderCode) {
        this.reorderCode = ValueStrategy.value(reorderCode);
        return getThis();
    }

    /**
     * Sets the expected reorder description.
     *
     * @param reorderDescription the reorder description
     * @return this
     */
    public B reorderDescription(String reorderDescription) {
        this.reorderDescription = ValueStrategy.value(reorderDescription);
        return getThis();
    }

    /**
     * Sets the expected package size.
     *
     * @param packageSize the package size
     * @return this
     */
    public B packageSize(int packageSize) {
        this.packageSize = ValueStrategy.value(packageSize);
        return getThis();
    }

    /**
     * Sets the expected package units.
     *
     * @param packageUnits the package units
     * @return this
     */
    public B packageUnits(String packageUnits) {
        this.packageUnits = ValueStrategy.value(packageUnits);
        return getThis();
    }

    /**
     * Sets the expected quantity.
     *
     * @param quantity the quantity
     * @return this
     */
    public B quantity(int quantity) {
        return quantity(BigDecimal.valueOf(quantity));
    }

    /**
     * Sets the expected quantity.
     *
     * @param quantity the quantity
     * @return this
     */
    public B quantity(BigDecimal quantity) {
        this.quantity = ValueStrategy.value(quantity);
        return getThis();
    }

    /**
     * Sets the expected unit price.
     *
     * @param unitPrice the unit price
     * @return this
     */
    public B unitPrice(int unitPrice) {
        return unitPrice(BigDecimal.valueOf(unitPrice));
    }

    /**
     * Sets the expected unit price.
     *
     * @param unitPrice the unit price
     * @return this
     */
    public B unitPrice(BigDecimal unitPrice) {
        this.unitPrice = ValueStrategy.value(unitPrice);
        return getThis();
    }

    /**
     * Sets the expected list price.
     *
     * @param listPrice the list price
     * @return this
     */
    public B listPrice(BigDecimal listPrice) {
        this.listPrice = ValueStrategy.value(listPrice);
        return getThis();
    }

    /**
     * Sets the expected tax.
     *
     * @param tax the tax
     * @return this
     */
    public B tax(BigDecimal tax) {
        this.tax = ValueStrategy.value(tax);
        return getThis();
    }

    /**
     * Sets the expected total.
     *
     * @param total the tax
     * @return this
     */
    public B total(BigDecimal total) {
        this.total = ValueStrategy.value(total);
        return getThis();
    }

    /**
     * Verifies there is an item that matches the criteria.
     *
     * @param items items
     * @return the matching item
     */
    public FinancialAct verify(List<FinancialAct> items) {
        FinancialAct item = items.stream().filter(act -> {
                    IMObjectBean bean = getBean(act);
                    return Objects.equals(product.getValue(), bean.getTargetRef("product"));
                }).findFirst()
                .orElse(null);
        if (item == null) {
            fail("Failed to find item for product=" + product);
        }
        verify(item);
        return item;
    }

    /**
     * Initialises this from an item.
     *
     * @param bean the bean wrapping the item
     */
    protected void initialise(IMObjectBean bean) {
        product(bean.getTargetRef("product"));
        reorderCode(bean.getString("reorderCode"));
        reorderDescription(bean.getString("reorderDescription"));
        packageSize(bean.getInt("packageSize"));
        packageUnits(bean.getString("packageUnits"));
        quantity(bean.getBigDecimal("quantity"));
        unitPrice(bean.getBigDecimal("unitPrice"));
        listPrice(bean.getBigDecimal("listPrice"));
        tax(bean.getBigDecimal("tax"));
        total(bean.getBigDecimal("total"));
    }

    /**
     * Verifies an object matches that expected.
     *
     * @param object the object to verify
     * @param bean   a bean wrapping the object
     */
    @Override
    protected void verify(FinancialAct object, IMObjectBean bean) {
        super.verify(object, bean);
        checkEquals(product, bean.getTargetRef("product"));
        checkEquals(reorderCode, bean.getString("reorderCode"));
        checkEquals(reorderDescription, bean.getString("reorderDescription"));
        checkEquals(packageSize, bean.getValue("packageSize"));
        checkEquals(packageUnits, bean.getString("packageUnits"));
        checkEquals(quantity, bean.getBigDecimal("quantity"));
        checkEquals(unitPrice, bean.getBigDecimal("unitPrice"));
        checkEquals(listPrice, bean.getBigDecimal("listPrice"));
        checkEquals(tax, bean.getBigDecimal("tax"));
        checkEquals(total, bean.getBigDecimal("total"));
    }
}
