/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.test.builder.patient;

import org.openvpms.archetype.rules.patient.PatientArchetypes;
import org.openvpms.component.math.WeightUnits;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.math.BigDecimal;
import java.util.Set;

/**
 * Builder for <em>act.patientWeight</em> acts for testing purposes.
 *
 * @author Tim Anderson
 */
public class TestWeightBuilder extends AbstractTestPatientActBuilder<Act, TestWeightBuilder> {

    /**
     * The weight.
     */
    private BigDecimal weight;

    /**
     * The weight units.
     */
    private WeightUnits units;

    /**
     * Constructs a {@link TestWeightBuilder}.
     *
     * @param service the archetype service
     */
    public TestWeightBuilder(ArchetypeService service) {
        super(PatientArchetypes.PATIENT_WEIGHT, Act.class, service);
    }

    /**
     * Sets the weight, in the default units.
     *
     * @param weight the weight
     * @return this
     */
    public TestWeightBuilder weight(int weight) {
        return weight(BigDecimal.valueOf(weight));
    }

    /**
     * Sets the weight, in the default units.
     *
     * @param weight the weight
     * @return this
     */
    public TestWeightBuilder weight(BigDecimal weight) {
        return weight(weight, null);
    }

    /**
     * Sets the weight, in the default units.
     *
     * @param weight the weight
     * @param units  the weight units
     * @return this
     */
    public TestWeightBuilder weight(int weight, WeightUnits units) {
        return weight(BigDecimal.valueOf(weight), units);
    }

    /**
     * Sets the weight.
     *
     * @param weight the weight
     * @param units  the weight units
     * @return this
     */
    public TestWeightBuilder weight(BigDecimal weight, WeightUnits units) {
        this.weight = weight;
        this.units = units;
        return this;
    }

    /**
     * Builds the object.
     *
     * @param object   the object
     * @param bean     a bean wrapping the object
     * @param toSave   objects to save, if the object is to be saved
     * @param toRemove the objects to remove
     */
    @Override
    protected void build(Act object, IMObjectBean bean, Set<IMObject> toSave, Set<Reference> toRemove) {
        super.build(object, bean, toSave, toRemove);
        if (weight != null) {
            bean.setValue("weight", weight);
        }
        if (units != null) {
            bean.setValue("units", units.toString());
        }
    }
}
