/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.supplier;

import org.openvpms.archetype.rules.stock.StockArchetypes;
import org.openvpms.component.business.service.archetype.ArchetypeServiceHelper;
import org.openvpms.component.business.service.archetype.helper.IMObjectBean;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.model.product.Product;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import static org.openvpms.archetype.test.TestHelper.create;
import static org.openvpms.archetype.test.TestHelper.save;


/**
 * Supplier test case helper methods.
 *
 * @author Tim Anderson
 */
public class SupplierTestHelper {

    /**
     * Package units for act items.
     */
    protected static final String PACKAGE_UNITS = "BOX";

    /**
     * Creates and saves a new stock location.
     *
     * @return a new stock location
     */
    public static Party createStockLocation() {
        Party stockLocation = create(StockArchetypes.STOCK_LOCATION, Party.class);
        stockLocation.setName("STOCK-LOCATION-" + stockLocation.hashCode());
        save(stockLocation);
        return stockLocation;
    }

    /**
     * Helper to create a POSTED <em>act.supplierOrder</em> and corresponding <em>act.supplierOrderItem</em>.
     *
     * @param amount        the act total
     * @param supplier      the supplier
     * @param stockLocation the stockLocation
     * @param product       the product
     * @return a list containing the order act and its item
     */
    public static List<FinancialAct> createOrder(BigDecimal amount, Party supplier, Party stockLocation,
                                                 Product product) {
        FinancialAct item = createOrderItem(product, BigDecimal.ONE, 1, PACKAGE_UNITS, amount, amount);
        return createOrder(supplier, stockLocation, item);
    }

    /**
     * Helper to create a POSTED <em>act.supplierOrder</em> with corresponding <em>act.supplierOrderItem</em>s.
     *
     * @param supplier      the supplier
     * @param stockLocation the stockLocation
     * @param items         the order items
     * @return a list containing the order act and its item
     */
    public static List<FinancialAct> createOrder(Party supplier, Party stockLocation, FinancialAct... items) {
        List<FinancialAct> result = new ArrayList<>();
        FinancialAct act = create(SupplierArchetypes.ORDER, FinancialAct.class);
        act.setStatus(OrderStatus.POSTED);
        result.add(act);

        IMObjectBean bean = new IMObjectBean(act);
        bean.setTarget("supplier", supplier);
        bean.setTarget("stockLocation", stockLocation);
        BigDecimal total = BigDecimal.ZERO;
        for (FinancialAct item : items) {
            ActRelationship relationship = (ActRelationship) bean.addTarget("items", item);
            item.addActRelationship(relationship);
            total = total.add(item.getTotal());
            result.add(item);
        }
        bean.setValue("amount", total);
        return result;
    }

    /**
     * Helper to create an order item.
     *
     * @param product      the product
     * @param quantity     the quantity
     * @param packageSize  the package size
     * @param packageUnits the package units
     * @param unitPrice    the unit price
     * @param listPrice    the list price
     * @return a new order item
     */
    public static FinancialAct createOrderItem(Product product, BigDecimal quantity, int packageSize,
                                               String packageUnits, BigDecimal unitPrice, BigDecimal listPrice) {
        FinancialAct item = create(SupplierArchetypes.ORDER_ITEM, FinancialAct.class);
        IMObjectBean bean = new IMObjectBean(item);
        bean.setTarget("product", product);
        item.setQuantity(quantity);
        bean.setValue("packageSize", packageSize);
        bean.setValue("packageUnits", packageUnits);
        bean.setValue("unitPrice", unitPrice);
        bean.setValue("listPrice", listPrice);
        ArchetypeServiceHelper.getArchetypeService().deriveValues(item);
        return item;
    }

}
