/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2021 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.statement;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes;
import org.openvpms.archetype.rules.finance.account.CustomerAccountQueryFactory;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.ArchetypeQueryException;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.IterableIMObjectQuery;
import org.openvpms.component.system.common.query.NodeConstraint;
import org.openvpms.component.system.common.query.NodeSelectConstraint;
import org.openvpms.component.system.common.query.NodeSortConstraint;
import org.openvpms.component.system.common.query.ObjectSet;
import org.openvpms.component.system.common.query.ObjectSetQueryIterator;
import org.openvpms.component.system.common.query.OrConstraint;
import org.openvpms.component.system.common.query.RelationalOp;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.CLOSING_BALANCE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.OPENING_BALANCE;


/**
 * Helper for performing statement act queries.
 *
 * @author Tim Anderson
 */
class StatementActHelper {


    public static class ActState {

        private final Date startTime;

        private final BigDecimal amount;

        private final boolean printed;

        public ActState(Date startTime, BigDecimal amount, boolean printed) {
            this.startTime = startTime;
            this.amount = amount;
            this.printed = printed;
        }

        public Date getStartTime() {
            return startTime;
        }

        public BigDecimal getAmount() {
            return amount;
        }

        public boolean isPrinted() {
            return printed;
        }
    }


    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * The short names to query. This contains all debit/credit parent acts,
     * and opening and closing balances.
     */
    private static final String[] SHORT_NAMES;

    /**
     * Charge short names.
     */
    private static final String[] CHARGE_SHORT_NAMES = {
            CustomerAccountArchetypes.INVOICE,
            CustomerAccountArchetypes.COUNTER,
            CustomerAccountArchetypes.CREDIT
    };

    /**
     * Start time node name.
     */
    private static final String START_TIME = "startTime";

    /**
     * Act start time.
     */
    private static final String ACT_START_TIME = "act.startTime";

    /**
     * Status node name.
     */
    private static final String STATUS = "status";

    static {
        List<String> shortNames = new ArrayList<>(CustomerAccountArchetypes.DEBITS_CREDITS);
        shortNames.add(CustomerAccountArchetypes.OPENING_BALANCE);
        shortNames.add(CustomerAccountArchetypes.CLOSING_BALANCE);
        SHORT_NAMES = shortNames.toArray(new String[0]);
    }

    /**
     * Constructs a {@link StatementActHelper}.
     *
     * @param service the archetype service
     */
    public StatementActHelper(IArchetypeService service) {
        this.service = service;
    }

    /**
     * Returns the timestamp for statement processing.
     *
     * @param statementDate the statement date
     * @return the date
     */
    public Date getStatementTimestamp(Date statementDate) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(statementDate);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * Determines if a customer has had end-of-period run on or after a
     * particular date.
     *
     * @param customer the customer
     * @param date     the date
     * @return {@code true} if end-of-period has been run on or after the date
     * @throws ArchetypeServiceException for any archetype service error
     */
    public boolean hasStatement(Party customer, Date date) {
        ActState state = getClosingBalanceAfter(customer, date);
        return (state != null);
    }

    /**
     * Returns the closing balance act for the specified statement date.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the closing balance for the statement date, or {@code null} if
     * none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public FinancialAct getClosingBalance(Party customer, Date statementDate) {
        ArchetypeQuery query = CustomerAccountQueryFactory.createQuery(
                customer, new String[]{CLOSING_BALANCE});
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(statementDate);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        query.add(new NodeConstraint(ACT_START_TIME, RelationalOp.GTE, calendar.getTime()));
        calendar.add(Calendar.DATE, 1);
        query.add(new NodeConstraint(ACT_START_TIME, RelationalOp.LT, calendar.getTime()));
        query.setMaxResults(1);
        Iterator<FinancialAct> iter = new IMObjectQueryIterator<>(service, query);
        return (iter.hasNext()) ? iter.next() : null;
    }

    /**
     * Returns all statement acts for a customer between the opening balance
     * prior to the specified date, and the corresponding closing balance
     * inclusive. Acts may have any status.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the statement acts
     * @throws ArchetypeServiceException for any archetype service error
     * @throws ArchetypeQueryException   for any archetype query error
     */
    public Iterable<Act> getActs(Party customer, Date statementDate) {
        Date open = getOpeningBalanceTimestamp(customer, statementDate);
        Date close = getClosingBalanceTimestamp(customer, statementDate, open);
        ArchetypeQuery query = createQuery(customer, open, close, true, null);
        return new IterableIMObjectQuery<>(service, query);
    }

    /**
     * Returns all POSTED statement acts between the opening and closing balance
     * timestamps. The result includes the opening balance, but excludes the
     * closing balance.
     *
     * @param customer                the customer
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @param closingBalanceTimestamp the closing balance timestamp. May be {@code null}
     * @param includeClosingBalance   if {@code true} includes the closing balance act, if present
     * @return the posted acts
     * @throws ArchetypeServiceException for any archetype service error
     * @throws ArchetypeQueryException   for any archetype query error
     */
    public Iterable<FinancialAct> getPostedActs(Party customer, Date openingBalanceTimestamp,
                                                Date closingBalanceTimestamp, boolean includeClosingBalance) {
        ArchetypeQuery query = createQuery(customer, openingBalanceTimestamp, closingBalanceTimestamp,
                                           includeClosingBalance, ActStatus.POSTED);
        return new IterableIMObjectQuery<>(service, query);
    }

    /**
     * Returns all COMPLETED charge act for a customer, between the
     * opening and closing balance timestamps.
     * If there is no closing balance timestamp, uses the statement date
     * timestamp.
     *
     * @param customer                the customer
     * @param statementDate           the statement date
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @param closingBalanceTimestamp the closing balance timestamp. May be {@code null}
     * @return any charge acts with COMPLETED status
     * @throws ArchetypeServiceException for any archetype service error
     * @throws ArchetypeQueryException   for any archetype query error
     */
    public Iterable<Act> getCompletedCharges(Party customer, Date statementDate, Date openingBalanceTimestamp,
                                             Date closingBalanceTimestamp) {
        if (closingBalanceTimestamp == null) {
            closingBalanceTimestamp = getStatementTimestamp(statementDate);
        }
        ArchetypeQuery query = createQuery(customer, CHARGE_SHORT_NAMES, openingBalanceTimestamp, false,
                                           closingBalanceTimestamp, false, ActStatus.COMPLETED);
        return new IterableIMObjectQuery<>(service, query);
    }

    /**
     * Returns all POSTED statement acts and COMPLETED charge acts for a
     * customer from the opening balance timestamp to the end of the statement
     * date. <p/>
     *
     * @param customer                the customer
     * @param statementDate           the date
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @return the statement acts
     * @throws ArchetypeServiceException for any archetype service error
     * @throws ArchetypeQueryException   for any archetype query error
     */
    public Iterable<FinancialAct> getPostedAndCompletedActs(
            Party customer, Date statementDate, Date openingBalanceTimestamp) {
        Date close = getStatementTimestamp(statementDate);
        ArchetypeQuery query = createQuery(customer, openingBalanceTimestamp, close, false, null);
        query.add(new OrConstraint()
                          .add(new NodeConstraint(STATUS, ActStatus.POSTED))
                          .add(new NodeConstraint(STATUS, ActStatus.COMPLETED)));
        return new IterableIMObjectQuery<>(service, query);
    }

    /**
     * Returns the opening balance timestamp for a customer and statement date.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the opening balance, or {@code null} if none is found
     */
    public Date getOpeningBalanceTimestamp(Party customer, Date statementDate) {
        StatementActHelper.ActState state = getOpeningBalanceState(customer, statementDate);
        return (state != null) ? state.getStartTime() : null;
    }

    /**
     * Returns the closing balance timestamp for a customer relative to a
     * statement date and opening balance timestamp.
     *
     * @param customer                the customer
     * @param statementDate           the statement date
     * @param openingBalanceTimestamp the opening balance timestamp. May be
     *                                {@code null}
     * @return the closing balance timestamp, or {@code null} if none is found
     */
    public Date getClosingBalanceTimestamp(Party customer, Date statementDate, Date openingBalanceTimestamp) {
        ActState state = getClosingBalanceState(customer, statementDate, openingBalanceTimestamp);
        return (state != null) ? state.getStartTime() : null;
    }

    /**
     * Returns the opening balance act state for a customer prior to  the
     * specified statement date.
     *
     * @param customer      the customer
     * @param statementDate the statement date
     * @return the opening balance state
     */
    public ActState getOpeningBalanceState(Party customer, Date statementDate) {
        return getActState(OPENING_BALANCE, customer, statementDate, RelationalOp.LT, false);
    }

    /**
     * Returns the closing balance act state for a customer relative to a
     * statement date and opening balance timestamp.
     *
     * @param customer                the customer
     * @param statementDate           the statement date
     * @param openingBalanceTimestamp the opening balance timestamp. May be
     *                                {@code null}
     * @return the closing balance state, or {@code null} if none is found
     */
    public ActState getClosingBalanceState(Party customer, Date statementDate, Date openingBalanceTimestamp) {
        ActState result;
        if (openingBalanceTimestamp == null) {
            result = getClosingBalanceBefore(customer, statementDate);
            if (result == null) {
                result = getClosingBalanceAfter(customer, statementDate);
            }
        } else {
            result = getClosingBalanceAfter(customer, openingBalanceTimestamp);
        }
        return result;
    }

    /**
     * Determines if there is any account activity between the specified
     * timetamps for a customer.
     *
     * @param customer                the customer
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @param closingBalanceTimestamp the closing balance timestamp. May be {@code null}
     * @return {@code true} if there is any account activity between the specified times
     */
    public boolean hasAccountActivity(Party customer, Date openingBalanceTimestamp, Date closingBalanceTimestamp) {
        ArchetypeQuery query = createQuery(customer, SHORT_NAMES, openingBalanceTimestamp, false,
                                           closingBalanceTimestamp, false, null);
        query.add(new NodeSelectConstraint(ACT_START_TIME));
        query.setMaxResults(1);
        Iterator<ObjectSet> iter = new ObjectSetQueryIterator(service, query);
        return iter.hasNext();
    }

    /**
     * Returns the state of a customer act whose startTime is before/after
     * the specified date, depending on the supplied operator.
     *
     * @param shortName     the act short name
     * @param customer      the customer
     * @param date          the date
     * @param operator      the operator
     * @param sortAscending if {@code true} sort acts on ascending startTime; otherwise sort them on descending
     *                      startTime
     * @return the state, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    public ActState getActState(String shortName, Party customer, Date date, RelationalOp operator,
                                boolean sortAscending) {
        ArchetypeQuery query = CustomerAccountQueryFactory.createQuery(customer, new String[]{shortName});
        if (date != null) {
            query.add(new NodeConstraint(ACT_START_TIME, operator, date));
        }
        query.add(new NodeSelectConstraint(ACT_START_TIME));
        query.add(new NodeSelectConstraint("act.amount"));
        query.add(new NodeSelectConstraint("act.credit"));
        query.add(new NodeSelectConstraint("act.printed"));
        query.add(new NodeSortConstraint(START_TIME, sortAscending));
        query.setMaxResults(1);
        ObjectSetQueryIterator iter = new ObjectSetQueryIterator(service,
                                                                 query);
        if (iter.hasNext()) {
            ObjectSet set = iter.next();
            Date startTime = (Date) set.get(ACT_START_TIME);
            BigDecimal amount = (BigDecimal) set.get("act.amount");
            boolean credit = (Boolean) set.get("act.credit");
            boolean printed = (Boolean) set.get("act.printed");
            if (credit) {
                amount = amount.negate();
            }
            return new ActState(startTime, amount, printed);
        }
        return null;
    }

    /**
     * Returns the state of the first
     * {@code act.customerAccountClosingBalance} for a customer, before the
     * specified timetamp.
     *
     * @param customer  the customer
     * @param timestamp the timestamp
     * @return the closing balance act startTime, or {@code null} if none is
     * found
     * @throws ArchetypeServiceException for any archetype service error
     */
    private ActState getClosingBalanceBefore(Party customer, Date timestamp) {
        return getActState(CLOSING_BALANCE, customer, timestamp, RelationalOp.LT, false);
    }

    /**
     * Returns the state of the first {@code act.customerAccountClosingBalance} for a customer, after
     * the specified timestamp.
     *
     * @param customer the customer
     * @param timetamp the timestamp
     * @return the closing balance act state, or {@code null} if none is found
     * @throws ArchetypeServiceException for any archetype service error
     */
    private ActState getClosingBalanceAfter(Party customer, Date timetamp) {
        return getActState(CLOSING_BALANCE, customer, timetamp, RelationalOp.GT, true);
    }

    /**
     * Helper to create a query for all account act types between
     * the opening and closing balance timetamps. This includes the opening
     * balance act, and optionally the closing balance.
     *
     * @param customer                the customer
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @param closingBalanceTimestamp the closing balance timestamp. May be {@code null}
     * @param includeClosingBalance   if {@code true} includes the closing balance act, if present
     * @param status                  the status. May be {@code null}
     * @return a new query
     * @throws ArchetypeServiceException for any archetype service error
     */
    private ArchetypeQuery createQuery(Party customer, Date openingBalanceTimestamp, Date closingBalanceTimestamp,
                                       boolean includeClosingBalance, String status) {
        return createQuery(customer, SHORT_NAMES, openingBalanceTimestamp, true, closingBalanceTimestamp,
                           includeClosingBalance, status);
    }

    /**
     * Helper to creates a query for all account act types between the opening and closing balance timestamps. The query
     * includes the opening balance.
     *
     * @param customer                the customer
     * @param shortNames              the account act short names
     * @param openingBalanceTimestamp the opening balance timestamp. May be {@code null}
     * @param includeOpeningBalance   if {@code true} includes the opening balance act, if present
     * @param closingBalanceTimestamp the closing balance timestamp. May be {@code null}
     * @param includeClosingBalance   if {@code true} includes the closing balance act, if present
     * @param status                  the status. May be {@code null}
     * @return a new query
     * @throws ArchetypeServiceException for any archetype service error
     */
    private ArchetypeQuery createQuery(Party customer, String[] shortNames,
                                       Date openingBalanceTimestamp, boolean includeOpeningBalance,
                                       Date closingBalanceTimestamp, boolean includeClosingBalance,
                                       String status) {
        ArchetypeQuery query = CustomerAccountQueryFactory.createQuery(customer, shortNames);
        if (openingBalanceTimestamp != null) {
            RelationalOp op = (includeOpeningBalance) ? RelationalOp.GTE : RelationalOp.GT;
            query.add(new NodeConstraint(START_TIME, op, openingBalanceTimestamp));
        }
        if (closingBalanceTimestamp != null) {
            RelationalOp op = (includeClosingBalance) ? RelationalOp.LTE : RelationalOp.LT;
            query.add(new NodeConstraint(START_TIME, op, closingBalanceTimestamp));
        }
        if (status != null) {
            query.add(Constraints.eq(STATUS, status));
        }
        query.add(Constraints.sort(START_TIME));
        query.add(Constraints.sort("id"));
        return query;
    }

}
