/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.apache.commons.lang3.StringUtils;
import org.openvpms.component.business.service.archetype.helper.IMObjectCopier;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.service.archetype.ArchetypeService;

import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static org.openvpms.archetype.rules.act.ActStatus.IN_PROGRESS;
import static org.openvpms.archetype.rules.act.ActStatus.POSTED;

/**
 * Reverses {@link FinancialAct}s.
 *
 * @author Tim Anderson
 */
class ReversalRules {

    /**
     * The reversal of an act.
     */
    public static class Reversal {

        private final FinancialAct reversal;

        private final Set<IMObject> changes;

        public Reversal(FinancialAct reversal, Set<IMObject> changes) {
            this.reversal = reversal;
            this.changes = changes;
        }

        /**
         * Returns the reversal.
         *
         * @return the reversal
         */
        public FinancialAct getReversal() {
            return reversal;
        }

        /**
         * Changes that need to be saved.
         *
         * @return the changes
         */
        public Set<IMObject> getChanges() {
            return changes;
        }
    }

    /**
     * The archetype service.
     */
    private final ArchetypeService service;

    /**
     * Hide node name.
     */
    private static final String HIDE = "hide";

    /**
     * Notes node name.
     */
    private static final String NOTES = "notes";

    /**
     * Reference node name.
     */
    private static final String REFERENCE = "reference";

    /**
     * Reversal relationship node name.
     */
    private static final String REVERSAL = "reversal";

    /**
     * Reverses relationship node name.
     */
    private static final String REVERSES = "reverses";


    /**
     * Constructs a {@link ReversalRules}.
     *
     * @param service the archetype service
     */
    public ReversalRules(ArchetypeService service) {
        this.service = service;
    }

    /**
     * Determines if an act has been reversed.
     *
     * @param act the act
     * @return {@code true} if the act has been reversed
     */
    public boolean isReversed(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.hasNode(REVERSAL) && !bean.getValues(REVERSAL).isEmpty();
    }

    /**
     * Determines if an act is a reversal of another.
     *
     * @param act the act
     * @return {@code true} if the act is a reversal
     */
    public boolean isReversal(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.hasNode(REVERSES) && !bean.getValues(REVERSES).isEmpty();
    }

    /**
     * Reverses an act.
     *
     * @param act                     the act to reverse
     * @param startTime               the start time of the reversal
     * @param reference               the reference. If {@code null}, the act identifier will be used
     * @param notes                   notes indicating the reason for the reversal, to set the 'notes' node if the
     *                                act has one. May be {@code null}
     * @param post                    if {@code true}, the reversal will have POSTED status, else it will have
     *                                IN_PROGRESS status. If posted, the act may not have APPROVED EFT items nor
     *                                payment processor items, as these need to be reversed via a 3rd-party service
     *                                prior to the reversal being POSTED.
     * @param hide                    if {@code true}, hide the reversal iff the act being reversed isn't already
     *                                hidden
     * @return the reversal of {@code act}
     */
    public Reversal reverse(FinancialAct act, Date startTime, String reference, String notes, boolean post,
                            boolean hide) {
        if (!POSTED.equals(act.getStatus())) {
            throw new IllegalStateException("Cannot reverse act with status " + act.getStatus());
        }
        IMObjectBean original = service.getBean(act);
        if (!original.getValues(REVERSAL).isEmpty()) {
            throw new IllegalStateException("Act=" + act.getId() + " has already been reversed");
        }
        IMObjectCopier copier = new IMObjectCopier(new CustomerActReversalHandler(act, post), service);
        List<IMObject> copies = copier.apply(act);
        FinancialAct reversal = (FinancialAct) copies.get(0);
        IMObjectBean bean = service.getBean(reversal);
        bean.setValue(REFERENCE, !StringUtils.isEmpty(reference) ? reference : act.getId());
        bean.setValue(NOTES, notes);
        if (post) {
            reversal.setStatus(POSTED);
        } else {
            reversal.setStatus(IN_PROGRESS);
        }
        reversal.setActivityStartTime(startTime);
        original.addTarget(REVERSAL, reversal, REVERSES);

        if (hide && !original.getBoolean(HIDE)) {
            bean.setValue(HIDE, true);
            original.setValue(HIDE, true);
        }
        return new Reversal(reversal, new HashSet<>(copies));
    }
}