/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2023 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.junit.Test;
import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.finance.eft.EFTPOSTransactionStatus;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.archetype.test.ArchetypeServiceTest;
import org.openvpms.archetype.test.builder.customer.TestCustomerFactory;
import org.openvpms.archetype.test.builder.customer.account.TestCustomerAccountFactory;
import org.openvpms.archetype.test.builder.paymentprocessor.TestPaymentProcessorFactory;
import org.openvpms.archetype.test.builder.practice.TestPracticeFactory;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.party.Party;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;

import static java.math.BigDecimal.TEN;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * Tests the <em>act.customerAccount*</em> archetypes.
 *
 * @author Tim Anderson
 */
public class CustomerAccountActTestCase extends ArchetypeServiceTest {

    /**
     * The customer factory.
     */
    @Autowired
    private TestCustomerFactory customerFactory;

    /**
     * The account factory.
     */
    @Autowired
    private TestCustomerAccountFactory accountFactory;

    /**
     * The payment processor factory.
     */
    @Autowired
    private TestPaymentProcessorFactory paymentProcessorFactory;

    /**
     * The practice factory.
     */
    @Autowired
    private TestPracticeFactory practiceFactory;

    /**
     * Verifies that when a payment is deleted, the payment items are also deleted, but not any EFTPOS or
     * payment processor acts.
     */
    @Test
    public void testDeletePayment() {
        Party customer = customerFactory.createCustomer();
        Entity terminal = practiceFactory.createEFTPOSTerminal();
        Party location = practiceFactory.createLocation();
        FinancialAct eftposTransaction = accountFactory.newEFTPOSPayment()
                .customer(customer)
                .amount(TEN)
                .terminal(terminal)
                .location(location)
                .status(EFTPOSTransactionStatus.PENDING)
                .build();

        FinancialAct eftposItem = accountFactory.newEFTPaymentItem()
                .addTransaction(eftposTransaction)
                .amount(TEN)
                .build(false);

        FinancialAct ppTransaction = accountFactory.newPaymentProcessorPayment()
                .customer(customer)
                .amount(TEN)
                .paymentProcessor(paymentProcessorFactory.createPaymentProcessor())
                .location(location)
                .status(PaymentProcessorTransactionStatus.PENDING)
                .build();
        FinancialAct ppItem = accountFactory.newPaymentProcessorPaymentItem()
                .processor(paymentProcessorFactory.createPaymentProcessor())
                .addTransaction(ppTransaction)
                .amount(TEN)
                .build(false);

        FinancialAct payment = accountFactory.newPayment()
                .customer(customer)
                .till(practiceFactory.createTill())
                .add(eftposItem, ppItem)
                .status(FinancialActStatus.IN_PROGRESS)
                .build();

        assertNotNull(get(payment));
        assertNotNull(get(eftposItem));
        assertNotNull(get(eftposTransaction));
        assertNotNull(get(ppItem));
        assertNotNull(get(ppTransaction));

        // verify the EFTPOS transaction is linked to the EFTPOS item
        List<IMObject> eft = getBean(eftposItem).getTargets("transactions");
        assertEquals(1, eft.size());
        assertEquals(eftposTransaction, eft.get(0));

        // verify the payment processor transaction is linked to the payment processor item
        List<IMObject> ppTransactions = getBean(ppItem).getTargets("transactions");
        assertEquals(1, ppTransactions.size());
        assertEquals(ppTransaction, ppTransactions.get(0));

        // verify that deleting the payment deletes the items, but not the transactions
        remove(payment);
        assertNull(get(eftposItem));
        assertNotNull(get(eftposTransaction));
        assertNull(get(ppItem));
        assertNotNull(get(ppTransaction));
    }

    /**
     * Verifies that when a refund is deleted, the refund items are also deleted, but not any EFTPOS or payment
     * processor acts.
     */
    @Test
    public void testDeleteRefund() {
        Party customer = customerFactory.createCustomer();
        Entity terminal = practiceFactory.createEFTPOSTerminal();
        Party location = practiceFactory.createLocation();
        FinancialAct eftposTransaction = accountFactory.newEFTPOSRefund()
                .customer(customer)
                .amount(TEN)
                .terminal(terminal)
                .location(location)
                .status(EFTPOSTransactionStatus.PENDING)
                .build();

        FinancialAct eftposItem = accountFactory.newEFTRefundItem()
                .addTransaction(eftposTransaction)
                .amount(TEN)
                .build(false);

        Entity paymentProcessor = paymentProcessorFactory.createPaymentProcessor();
        FinancialAct ppTransaction = accountFactory.newPaymentProcessorRefund()
                .customer(customer)
                .amount(TEN)
                .paymentProcessor(paymentProcessor)
                .location(location)
                .status(PaymentProcessorTransactionStatus.PENDING)
                .build();

        FinancialAct ppItem = accountFactory.newPaymentProcessorRefundItem()
                .processor(paymentProcessor)
                .addTransaction(ppTransaction)
                .amount(TEN)
                .build(false);

        FinancialAct refund = accountFactory.newRefund()
                .customer(customer)
                .till(practiceFactory.createTill())
                .add(eftposItem, ppItem)
                .status(FinancialActStatus.IN_PROGRESS)
                .build();

        assertNotNull(get(refund));
        assertNotNull(get(eftposItem));
        assertNotNull(get(eftposTransaction));
        assertNotNull(get(ppItem));
        assertNotNull(get(ppTransaction));

        // verify the EFTPOS transaction is linked to the EFT item
        List<IMObject> eft = getBean(eftposItem).getTargets("transactions");
        assertEquals(1, eft.size());
        assertEquals(eftposTransaction, eft.get(0));

        // verify the payment processor transaction is linked to the payment processor item
        List<IMObject> ppTransactions = getBean(ppItem).getTargets("transactions");
        assertEquals(1, ppTransactions.size());
        assertEquals(ppTransaction, ppTransactions.get(0));

        // verify that deleting the refund deletes the items, but not the transactions
        remove(refund);
        assertNull(get(eftposItem));
        assertNotNull(get(eftposTransaction));
        assertNull(get(ppItem));
        assertNotNull(get(ppTransaction));
    }

}
