/*
 * Version: 1.0
 *
 * The contents of this file are subject to the OpenVPMS License Version
 * 1.0 (the 'License'); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.openvpms.org/license/
 *
 * Software distributed under the License is distributed on an 'AS IS' basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Copyright 2025 (C) OpenVPMS Ltd. All Rights Reserved.
 */

package org.openvpms.archetype.rules.finance.account;

import org.openvpms.archetype.rules.act.ActStatus;
import org.openvpms.archetype.rules.act.FinancialActStatus;
import org.openvpms.archetype.rules.finance.account.ReversalRules.Reversal;
import org.openvpms.archetype.rules.finance.eft.EFTPOSTransactionStatus;
import org.openvpms.archetype.rules.finance.paymentprocessor.PaymentProcessorTransactionStatus;
import org.openvpms.archetype.rules.finance.tax.CustomerTaxRules;
import org.openvpms.archetype.rules.finance.till.TillArchetypes;
import org.openvpms.archetype.rules.finance.till.TillBalanceRules;
import org.openvpms.archetype.rules.finance.till.TillBalanceStatus;
import org.openvpms.archetype.rules.math.MathRules;
import org.openvpms.archetype.rules.util.DateRules;
import org.openvpms.archetype.rules.util.DateUnits;
import org.openvpms.component.business.service.archetype.ArchetypeServiceException;
import org.openvpms.component.business.service.archetype.IArchetypeService;
import org.openvpms.component.business.service.archetype.helper.DescriptorHelper;
import org.openvpms.component.business.service.archetype.rule.IArchetypeRuleService;
import org.openvpms.component.model.act.Act;
import org.openvpms.component.model.act.ActRelationship;
import org.openvpms.component.model.act.FinancialAct;
import org.openvpms.component.model.bean.IMObjectBean;
import org.openvpms.component.model.bean.Policies;
import org.openvpms.component.model.bean.Predicates;
import org.openvpms.component.model.entity.Entity;
import org.openvpms.component.model.lookup.Lookup;
import org.openvpms.component.model.object.IMObject;
import org.openvpms.component.model.object.Reference;
import org.openvpms.component.model.party.Party;
import org.openvpms.component.query.criteria.CriteriaBuilder;
import org.openvpms.component.query.criteria.CriteriaQuery;
import org.openvpms.component.query.criteria.Root;
import org.openvpms.component.system.common.query.AndConstraint;
import org.openvpms.component.system.common.query.ArchetypeQuery;
import org.openvpms.component.system.common.query.Constraints;
import org.openvpms.component.system.common.query.IMObjectQueryIterator;
import org.openvpms.component.system.common.query.NodeConstraint;
import org.openvpms.component.system.common.query.ObjectSetQueryIterator;
import org.openvpms.component.system.common.query.RelationalOp;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.openvpms.archetype.rules.act.ActStatus.IN_PROGRESS;
import static org.openvpms.archetype.rules.act.ActStatus.POSTED;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.DEBITS;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.DEBITS_CREDITS;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.OPENING_BALANCE;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_OTHER;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.PAYMENT_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_EFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountArchetypes.REFUND_PP;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountRuleException.ErrorCode.AlreadyPosted;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountRuleException.ErrorCode.CannotPostAct;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountRuleException.ErrorCode.CannotPostWithOutstandingEFT;
import static org.openvpms.archetype.rules.finance.account.CustomerAccountRuleException.ErrorCode.CannotPostWithOutstandingPaymentProcessorItem;


/**
 * Customer account rules.
 *
 * @author Tim Anderson
 */
public class CustomerAccountRules {

    /**
     * The archetype service.
     */
    private final IArchetypeService service;

    /**
     * The rule based archetype service.
     */
    private final IArchetypeRuleService ruleService;

    /**
     * The customer balance updater.
     */
    private final CustomerBalanceUpdater updater;

    /**
     * The transaction manager.
     */
    private final PlatformTransactionManager transactionManager;

    /**
     * Balance calculator.
     */
    private final BalanceCalculator calculator;

    /**
     * The reversal rules.
     */
    private final ReversalRules reversalRules;

    /**
     * Id node name.
     */
    private static final String ID = "id";

    /**
     * Act start time node name.
     */
    private static final String START_TIME = "startTime";

    /**
     * Status node name.
     */
    private static final String STATUS = "status";

    /**
     * Customer account type node name.
     */
    private static final String TYPE = "type";

    /**
     * Hide node name.
     */
    private static final String HIDE = "hide";

    /**
     * Notes node name.
     */
    private static final String NOTES = "notes";

    /**
     * Customer node name.
     */
    private static final String CUSTOMER = "customer";

    /**
     * Items relationship node name.
     */
    private static final String ITEMS = "items";


    /**
     * Constructs a {@link CustomerAccountRules}.
     *
     * @param service            the archetype service
     * @param ruleService        the rule based archetype service
     * @param updater            the customer balance updater
     * @param transactionManager the transaction manager
     */
    public CustomerAccountRules(IArchetypeService service, IArchetypeRuleService ruleService,
                                CustomerBalanceUpdater updater, PlatformTransactionManager transactionManager) {
        // NOTE: need an IArchetypeRuleService as the reverse() methods need to fire rules to update the balance
        if (service instanceof IArchetypeRuleService) {
            throw new IllegalArgumentException("Argument 'service' should not implement IArchetypeRuleService");
        }
        this.service = service;
        this.ruleService = ruleService;
        this.updater = updater;
        this.transactionManager = transactionManager;
        calculator = new BalanceCalculator(service);
        reversalRules = new ReversalRules(service);
    }

    /**
     * Calculates a charge/estimate total using to two decimal places.
     * <br/>
     * If the quantity is:
     * <ul>
     *     <li>+ve: {@code fixedPrice + unitPrice * quantity - discount}</li>
     *     <li>-ve: {@code -(fixedPrice + unitPrice * abs(quantity) - discount)}</li>
     *     <li>zero: {@code 0}</li>
     * </ul>
     *
     * @param fixedPrice the fixed price
     * @param unitPrice  the unit price
     * @param quantity   the quantity
     * @param discount   the discount
     * @return the total
     */
    public BigDecimal calculateTotal(BigDecimal fixedPrice, BigDecimal unitPrice, BigDecimal quantity,
                                     BigDecimal discount) {
        return MathRules.calculateTotal(fixedPrice, unitPrice, quantity, discount, 2);
    }

    /**
     * Returns the opening balance before the specified date.
     *
     * @param date the date
     * @return the opening balance, or {@code null} if none is found
     */
    public FinancialAct getOpeningBalanceBefore(Party customer, Date date) {
        ArchetypeQuery query = CustomerAccountQueryFactory.createQuery(customer, OPENING_BALANCE);
        query.add(Constraints.lt(START_TIME, date));
        query.add(Constraints.sort(START_TIME, false));
        query.add(Constraints.sort(ID, false));
        query.setMaxResults(1);
        Iterator<FinancialAct> iterator = new IMObjectQueryIterator<>(service, query);
        return iterator.hasNext() ? iterator.next() : null;
    }

    /**
     * Returns the opening balance after the specified date.
     *
     * @param date the date
     * @return the opening balance, or {@code null} if none is found
     */
    public FinancialAct getOpeningBalanceAfter(Party customer, Date date) {
        ArchetypeQuery query = CustomerAccountQueryFactory.createQuery(customer, OPENING_BALANCE);
        query.add(Constraints.gt(START_TIME, date));
        query.add(Constraints.sort(START_TIME, true));
        query.add(Constraints.sort(ID, false));
        query.setMaxResults(1);
        Iterator<FinancialAct> iterator = new IMObjectQueryIterator<>(service, query);
        return iterator.hasNext() ? iterator.next() : null;
    }

    /**
     * Creates a new opening balance for a customer.
     *
     * @param customer the customer
     * @param date     the act date
     * @param amount   the amount. May be negative
     * @return a new opening balance
     */
    public FinancialAct createOpeningBalance(Party customer, Date date, BigDecimal amount) {
        return createBalance(OPENING_BALANCE, customer, date, amount);
    }

    /**
     * Creates a new closing balance for a customer.
     *
     * @param customer the customer
     * @param date     the act date
     * @param amount   the amount. May be negative
     * @return a new closing balance
     */
    public FinancialAct createClosingBalance(Party customer, Date date, BigDecimal amount) {
        return createBalance(CustomerAccountArchetypes.CLOSING_BALANCE, customer, date, amount);
    }

    /**
     * Calculates the outstanding balance for a customer.
     *
     * @param customer the customer
     * @return the balance
     * @throws ArchetypeServiceException for any archetype service error
     */
    public BigDecimal getBalance(Party customer) {
        return calculator.getBalance(customer);
    }

    /**
     * Calculates the outstanding balance for a customer, incorporating acts
     * up to the specified date.
     *
     * @param customer the customer
     * @param date     the date
     * @return the balance
     * @throws ArchetypeServiceException for any archetype service error
     */
    public BigDecimal getBalance(Party customer, Date date) {
        return calculator.getBalance(customer, date);
    }

    /**
     * Calculates the balance for a customer for all POSTED acts between two times, inclusive.
     *
     * @param customer       the customer
     * @param from           the from-time. If {@code null}, indicates that the time is unbounded
     * @param to             the to-time. If {@code null}, indicates that the time is unbounded
     * @param openingBalance the opening balance
     * @return the balance
     */
    public BigDecimal getBalance(Party customer, Date from, Date to, BigDecimal openingBalance) {
        return getBalance(customer, from, to, true, openingBalance);
    }

    /**
     * Calculates the balance for a customer for all POSTED acts between two times.
     *
     * @param customer       the customer
     * @param from           the from-time. If {@code null}, indicates that the time is unbounded
     * @param to             the to-time. If {@code null}, indicates that the time is unbounded
     * @param inclusive      if {@code true}, include acts {@code <= to} else include acts {@code < to}
     * @param openingBalance the opening balance
     * @return the balance
     */
    public BigDecimal getBalance(Party customer, Date from, Date to, boolean inclusive, BigDecimal openingBalance) {
        return calculator.getBalance(customer, from, to, inclusive, openingBalance);
    }

    /**
     * Calculates a definitive outstanding balance for a customer.
     * <p>
     * This sums total amounts for <em>all</em> POSTED acts associated with the
     * customer, rather than just using unallocated acts, and can be used
     * to detect account balance errors.
     *
     * @param customer the customer
     * @return the definitive balance
     * @throws ArchetypeServiceException    for any archetype service error
     * @throws CustomerAccountRuleException if an opening or closing balance
     *                                      is incorrect
     */
    public BigDecimal getDefinitiveBalance(Party customer) {
        return calculator.getDefinitiveBalance(customer);
    }

    /**
     * Calculates a new balance for a customer from the current outstanding
     * balance and a running total.
     * If the new balance is:
     * <ul>
     * <li>&lt; 0 returns 0.00 for payments, or -balance for refunds</li>
     * <li>&gt; 0 returns 0.00 for refunds</li>
     * </ul>
     *
     * @param customer the customer
     * @param total    the running total
     * @param payment  if {@code true} indicates the total is for a payment,
     *                 if {@code false} indicates it is for a refund
     * @return the new balance
     * @throws ArchetypeServiceException for any archetype service error
     */
    public BigDecimal getBalance(Party customer, BigDecimal total, boolean payment) {
        BigDecimal balance = getBalance(customer);
        BigDecimal result;
        if (payment) {
            result = balance.subtract(total);
        } else {
            result = balance.add(total);
        }
        if (result.signum() == -1) {
            result = (payment) ? BigDecimal.ZERO : result.negate();
        } else if (result.signum() == 1 && !payment) {
            result = BigDecimal.ZERO;
        }
        return result;
    }

    /**
     * Calculates the current overdue balance for a customer.
     * This is the sum of unallocated amounts in associated debits that have a
     * date less than the specified date less the overdue days.
     * The overdue days are specified in the customer's type node.
     * <p>
     * NOTE: this method may not be used to determine an historical overdue
     * balance. For this, use {@link #getOverdueBalance(Party, Date, Date)
     * getOverdueBalance(Party customer, Date date, Date overdueDate)}.
     *
     * @param customer the customer
     * @param date     the date
     * @return the overdue balance
     * @throws ArchetypeServiceException for any archetype service error
     */
    public BigDecimal getOverdueBalance(Party customer, Date date) {
        Date overdue = getOverdueDate(customer, date);
        return calculator.getOverdueBalance(customer, overdue);
    }

    /**
     * Calculates the overdue balance for a customer as of a particular date.
     * <p>
     * This sums any POSTED debits prior to <em>overdueDate</em> that had
     * not been fully allocated by credits as of <em>date</em>.
     *
     * @param customer    the customer
     * @param date        the date
     * @param overdueDate the date when amounts became overdue
     * @return the overdue balance
     */
    public BigDecimal getOverdueBalance(Party customer, Date date, Date overdueDate) {
        return calculator.getOverdueBalance(customer, date, overdueDate);
    }

    /**
     * Determines if a customer has an overdue balance within the nominated
     * day range past their standard terms.
     *
     * @param customer the customer
     * @param date     the date
     * @param from     the from day range
     * @param to       the to day range. Use {@code &lt;= 0} to indicate
     *                 all dates
     * @return {@code true} if the customer has an overdue balance within
     * the day range past their standard terms.
     */
    public boolean hasOverdueBalance(Party customer, Date date, int from, int to) {
        Date overdue = getOverdueDate(customer, date);
        Date overdueFrom = overdue;
        Date overdueTo = null;
        if (from > 0) {
            overdueFrom = DateRules.getDate(overdueFrom, -from, DateUnits.DAYS);
        }
        if (to > 0) {
            overdueTo = DateRules.getDate(overdue, -to, DateUnits.DAYS);
        }

        // query all overdue debit acts
        ArchetypeQuery query = CustomerAccountQueryFactory.createUnallocatedObjectSetQuery(customer, DEBITS);

        NodeConstraint fromStartTime = new NodeConstraint(START_TIME, RelationalOp.LT, overdueFrom);
        if (overdueTo == null) {
            query.add(fromStartTime);
        } else {
            NodeConstraint toStartTime = new NodeConstraint(START_TIME, RelationalOp.GT, overdueTo);
            AndConstraint and = new AndConstraint();
            and.add(fromStartTime);
            and.add(toStartTime);
            query.add(and);
        }
        query.setMaxResults(1);
        ObjectSetQueryIterator iterator = new ObjectSetQueryIterator(service, query);
        return iterator.hasNext();
    }

    /**
     * Returns the overdue date relative to the specified date, for a customer.
     *
     * @param customer the customer
     * @param date     the date
     * @return the overdue date
     */
    public Date getOverdueDate(Party customer, Date date) {
        IMObjectBean bean = service.getBean(customer);
        Date overdue = date;
        if (bean.hasNode(TYPE)) {
            List<Lookup> types = bean.getValues(TYPE, Lookup.class);
            if (!types.isEmpty()) {
                overdue = getOverdueDate(types.get(0), date);
            }
        }
        return overdue;
    }

    /**
     * Returns the overdue date relative to the specified date for a customer
     * type.
     *
     * @param type a <em>lookup.customerAccountType</em>
     * @param date the date
     * @return the overdue date
     */
    public Date getOverdueDate(Lookup type, Date date) {
        return new AccountType(type, service).getOverdueDate(date);
    }

    /**
     * Calculates the sum of all unbilled charge acts for a customer.
     *
     * @param customer the customer
     * @return the unbilled amount
     * @throws ArchetypeServiceException for any archetype service error
     */
    public BigDecimal getUnbilledAmount(Party customer) {
        return calculator.getUnbilledAmount(customer);
    }

    /**
     * Determines if an act has been reversed.
     *
     * @param act the act
     * @return {@code true} if the act has been reversed
     */
    public boolean isReversed(FinancialAct act) {
        return reversalRules.isReversed(act);
    }

    /**
     * Determines if an act is a reversal of another.
     *
     * @param act the act
     * @return {@code true} if the act is a reversal
     */
    public boolean isReversal(FinancialAct act) {
        return reversalRules.isReversal(act);
    }

    /**
     * Reverses an act.
     *
     * @param act       the act to reverse
     * @param startTime the start time of the reversal
     * @return the reversal of {@code act}
     * @throws ArchetypeServiceException for any archetype service error
     */
    public FinancialAct reverse(FinancialAct act, Date startTime) {
        return reverse(act, startTime, null, null, false);
    }

    /**
     * Reverses an act.
     * <p>
     * If the act to be reversed is an invoice, charge items and medication acts will be unlinked from patient history.
     * Reminders and investigations will be retained.
     *
     * @param act       the act to reverse
     * @param startTime the start time of the reversal
     * @param notes     notes indicating the reason for the reversal, to set the 'notes' node if the act has one.
     *                  May be {@code null}
     * @param reference the reference. If {@code null}, the act identifier will be used
     * @param hide      if {@code true}, hide the reversal iff the act being reversed isn't already hidden
     * @return the reversal of {@code act}
     * @throws ArchetypeServiceException for any archetype service error
     */
    public FinancialAct reverse(FinancialAct act, Date startTime, String notes, String reference, boolean hide) {
        return reverse(act, startTime, notes, reference, hide, null);
    }

    /**
     * Reverses an act.
     * <p>
     * If the act to be reversed is an invoice, charge items and medication acts will be unlinked from patient history.
     * Reminders and investigations will be retained.
     * <p/>
     * The reversal will be allocated to the act, where possible.
     * <p/>
     * NOTE: this does not support reversals of <em>act.customerAccountPaymentPP</em> or
     * <em>act.customerAccountRefundPP</em> as reversals of these need to be managed via their associated
     * payment processor services.
     *
     * @param act         the act to reverse
     * @param startTime   the start time of the reversal
     * @param notes       notes indicating the reason for the reversal, to set the 'notes' node if the act has one.
     *                    May be {@code null}
     * @param reference   the reference. If {@code null}, the act identifier will be used
     * @param hide        if {@code true}, hide the reversal iff the act being reversed isn't already hidden
     * @param tillBalance the till balance to add the reversal to. Only applies to payments and refunds, and
     *                    IN_PROGRESS till balance acts. May be {@code null}
     * @return the reversal of {@code act}
     * @throws ArchetypeServiceException for any archetype service error
     */
    public FinancialAct reverse(FinancialAct act, Date startTime, String notes, String reference, boolean hide,
                                FinancialAct tillBalance) {
        TransactionTemplate template = new TransactionTemplate(transactionManager);
        return template.execute(status -> {
            Reversal reversal = reversalRules.reverse(act, startTime, reference, notes, false, hide);
            Set<IMObject> changes = reversal.getChanges();

            // move allocations from the act to the reversal
            FinancialAct reversalAct = reversal.getReversal();
            Collection<FinancialAct> allocationChanges = updater.moveAllocations(act, reversalAct);
            changes.addAll(allocationChanges);

            boolean updateBalance = tillBalance != null && reversalAct.isA(PAYMENT, REFUND);
            TillBalanceRules rules = (updateBalance) ? new TillBalanceRules(service) : null;
            if (updateBalance) {
                List<Act> balanceChanges = rules.addToBalance(reversalAct, tillBalance);
                changes.addAll(balanceChanges);
            }

            // This smells. The original acts needs to be saved without using the rule based archetype service, to avoid
            // triggering rules. The other acts need to be saved with rules enabled, in order to update the balance.
            // TODO
            changes.remove(act);
            List<IMObject> noRules = new ArrayList<>();
            noRules.add(act);

            if (act.isA(CustomerAccountArchetypes.INVOICE)) {
                removeInvoiceFromPatientHistory(act, noRules);
            }
            service.save(noRules);
            ruleService.save(changes);

            // can only update the till balance when all the other objects have been saved
            if (rules != null) {
                rules.updateBalance(tillBalance);
            }
            return reversalAct;
        });
    }

    /**
     * Creates a reversal of a POSTED payment containing an <em>act.customerAccountPaymentPP</em> item.
     * <p/>
     * The reversal is an IN_PROGRESS refund with a corresponding <em>act.customerAccountRefundPP</em> item.<br/>
     * This must be submitted to the payment processor before the refund can be POSTED.
     * <p/>
     * The refund is saved.
     * <p/>
     * Reversals of refunds are not supported as it's not supported by the payment processors currently being used.
     *
     * @param payment the payment
     * @return the corresponding refund
     */
    public FinancialAct createPaymentProcessorReversal(FinancialAct payment) {
        if (!payment.isA(PAYMENT)) {
            throw new IllegalArgumentException("Argument 'payment' must be a " + PAYMENT + ", not a "
                                               + payment.getArchetype());
        }
        Reversal reversal = reversalRules.reverse(payment, new Date(), null, null, true, false);
        Set<IMObject> changes = reversal.getChanges();

        if (changes.stream().noneMatch(object -> object.isA(REFUND_PP))) {
            // not a refund of a payment processor payment. This will only occur if the method is misused.
            throw new IllegalStateException(REFUND_PP + " not found");
        }
        FinancialAct reversalAct = reversal.getReversal();
        reversalAct.setStatus(IN_PROGRESS);
        service.save(changes);
        return reversalAct;
    }

    /**
     * Returns the customer account archetype used to reverse the specified archetype.
     *
     * @param archetype the archetype
     * @return the corresponding archetype used in reversals, or {@code null} if none is found
     */
    public String getReversalArchetype(String archetype) {
        for (String[] pair : CustomerActReversalHandler.TYPE_MAP) {
            if (pair[0].equals(archetype)) {
                return pair[1];
            } else if (pair[1].equals(archetype)) {
                return pair[0];
            }
        }
        return null;
    }

    /**
     * Determines if the act has been fully allocated.
     *
     * @param act the act
     * @return {@code true} if the act has been full allocated
     */
    public boolean isAllocated(FinancialAct act) {
        return calculator.isAllocated(act);
    }

    /**
     * Sets the hidden state of a reversed/reversal act.
     *
     * @param act  the act
     * @param hide if {@code true}, hide the act in customer statements, else show it
     */
    public void setHidden(FinancialAct act, boolean hide) {
        if (canHide(act)) {
            IMObjectBean bean = service.getBean(act);
            // NOTE: must use non-rule based service to avoid balance recalculation
            if (hide != bean.getBoolean(HIDE)) {
                bean.setValue(HIDE, hide);
                // NOTE: must use non-rule based service to avoid balance recalculation
                service.save(act);
            }
        }
    }

    /**
     * Determines if an act is hidden in customer statements.
     *
     * @param act the act
     * @return {@code true} if the {@code hide} node is {@code true}
     */
    public boolean isHidden(Act act) {
        IMObjectBean bean = service.getBean(act);
        return bean.hasNode(HIDE) && bean.getBoolean(HIDE);
    }

    /**
     * Determines if an act can be hidden in customer statements.
     * <p>
     * Note that this doesn't take into account the hidden state of related acts.
     *
     * @param act the act
     * @return {@code true} if the act isn't hidden, and is reversed or a reversal
     */
    public boolean canHide(FinancialAct act) {
        return isReversed(act) || isReversal(act);
    }

    /**
     * Returns the latest {@code IN_PROGRESS} or {@code COMPLETED} invoice for a customer.
     * <p>
     * Invoices with {@code IN_PROGRESS} will be returned in preference to {@code COMPLETED} ones.
     *
     * @param customer the customer
     * @return the customer invoice, or {@code null} if none is found
     */
    public FinancialAct getInvoice(Party customer) {
        return getInvoice(customer.getObjectReference());
    }

    /**
     * Returns the latest {@code IN_PROGRESS} or {@code COMPLETED} invoice for a customer.
     * <p>
     * Invoices with {@code IN_PROGRESS} will be returned in preference to {@code COMPLETED} ones.
     *
     * @param customer the customer
     * @return the customer invoice, or {@code null} if none is found
     */
    public FinancialAct getInvoice(Reference customer) {
        return getCharge(CustomerAccountArchetypes.INVOICE, customer);
    }

    /**
     * Returns the latest {@code IN_PROGRESS} or {@code COMPLETED} credit for a customer.
     * <p>
     * Credits with {@code IN_PROGRESS} will be returned in preference to {@code COMPLETED} ones.
     *
     * @param customer the customer
     * @return the customer credit, or {@code null} if none is found
     */
    public FinancialAct getCredit(Party customer) {
        return getCredit(customer.getObjectReference());
    }

    /**
     * Returns the latest {@code IN_PROGRESS} or {@code COMPLETED} credit for a customer.
     * <p>
     * Credits with {@code IN_PROGRESS} will be returned in preference to {@code COMPLETED} ones.
     *
     * @param customer the customer
     * @return the customer credit, or {@code null} if none is found
     */
    public FinancialAct getCredit(Reference customer) {
        return getCharge(CustomerAccountArchetypes.CREDIT, customer);
    }

    /**
     * Determines if a customer has any account acts.
     *
     * @param customer the customer
     * @return {@code true} if the customer has any account acts
     */
    public boolean hasAccountActs(Party customer) {
        return updater.hasAccountActs(customer.getObjectReference());
    }

    /**
     * Creates a credit adjustment for a customer.
     *
     * @param customer the customer
     * @param total    the adjustment total
     * @param location the practice location. May be {@code null}
     * @param practice the practice, used to determine tax rates
     * @param notes    optional notes. May be {@code null}
     * @return a new credit adjustment
     */
    public FinancialAct createCreditAdjustment(Party customer, BigDecimal total, Party location, Party practice,
                                               String notes) {
        CustomerTaxRules taxRules = new CustomerTaxRules(practice, service);
        FinancialAct act = service.create(CustomerAccountArchetypes.CREDIT_ADJUST, FinancialAct.class);
        act.setTotal(total);
        act.setStatus(ActStatus.POSTED); // status is derived, but derived values aren't automatically populated.
        IMObjectBean bean = ruleService.getBean(act);

        bean.setTarget(CUSTOMER, customer);
        if (location != null) {
            bean.setTarget("location", location);
        }
        if (notes != null) {
            bean.setValue(NOTES, notes);
        }
        taxRules.calculateTax(act);
        return act;
    }

    /**
     * Creates customer payment, with a single 'Other' line item.
     * <p/>
     * The payment has {@link FinancialActStatus#POSTED} status, and is unsaved.
     *
     * @param customer    the customer
     * @param total       the payment total
     * @param till        the till
     * @param location    the practice location
     * @param paymentType the payment type. May be {@code null}
     * @param notes       the notes. May be {@code null}
     * @return the payment. The first act is the <em>act.customerAccountPayment</em>, the second the
     * <em>act.customerAccountPaymentOther</em>
     */
    public List<FinancialAct> createPaymentOther(Party customer, BigDecimal total, Entity till, Party location,
                                                 String paymentType, String notes) {
        FinancialAct act = service.create(PAYMENT, FinancialAct.class);
        FinancialAct item = service.create(PAYMENT_OTHER, FinancialAct.class);

        act.setStatus(POSTED);
        IMObjectBean bean = service.getBean(act);
        bean.setTarget(CUSTOMER, customer);
        bean.setValue("amount", total);
        bean.setTarget("till", till);
        bean.setTarget("location", location);
        if (notes != null) {
            bean.setValue(NOTES, notes);
        }

        IMObjectBean itemBean = service.getBean(item);
        itemBean.setValue("amount", total);
        if (paymentType != null) {
            itemBean.setValue("paymentType", paymentType);
        }
        ActRelationship relationship = (ActRelationship) bean.addTarget(ITEMS, item);
        item.addActRelationship(relationship);
        return Arrays.asList(act, item);
    }

    /**
     * Determines if a payment is associated with a till balance with {@code CLEARED} status.
     *
     * @param payment the payment
     * @return {@code true} if the payment is associated with a till balance, and the till balance has been cleared
     */
    public boolean hasClearedTillBalance(FinancialAct payment) {
        boolean result = false;
        IMObjectBean bean = service.getBean(payment);
        Reference balance = bean.getSourceRef("tillBalance");
        if (balance != null) {
            CriteriaBuilder builder = service.getCriteriaBuilder();
            CriteriaQuery<Long> query = builder.createQuery(Long.class);
            Root<Act> root = query.from(Act.class, TillArchetypes.TILL_BALANCE);
            query.select(root.get(ID));
            query.where(builder.equal(root.reference(), balance),
                        builder.notEqual(root.get(STATUS), TillBalanceStatus.CLEARED));
            if (service.createQuery(query).getFirstResult() == null) {
                result = true;
            }
        }
        return result;
    }

    /**
     * Determines if an act can be posted.
     *
     * @param act the act
     * @return {@code true} if the act can be posted, otherwise {@code false}
     */
    public boolean canPost(FinancialAct act) {
        return getPostStatus(act).canPost();
    }

    /**
     * Determines if a customer account act can be POSTED.
     *
     * @param act the act
     * @return the status of the act
     */
    public PostStatus getPostStatus(FinancialAct act) {
        PostStatus result = PostStatus.unsupported();
        if (act.isA(DEBITS_CREDITS)) {
            if (POSTED.equals(act.getStatus())) {
                result = PostStatus.posted();
            } else if (act.isA(PAYMENT, REFUND)) {
                ItemStatus eftStatus = getEFTPOSTransactionStatus(act, false);
                if (eftStatus.isIncomplete()) {
                    result = PostStatus.outstandingEFTPOSTransaction(eftStatus.getItem(), eftStatus.getTransaction());
                } else {
                    // EFT is complete, or there is no EFT transaction
                    ItemStatus ppsStatus = getPaymentProcessorTransactionStatus(act, false);
                    if (ppsStatus.isUnsubmitted()) {
                        result = PostStatus.unsubmittedPaymentProcessorTransaction(ppsStatus.getItem(),
                                                                                   ppsStatus.getTransaction());
                    } else if (ppsStatus.isIncomplete()) {
                        result = PostStatus.outstandingPaymentProcessorTransaction(ppsStatus.getItem(),
                                                                                   ppsStatus.getTransaction());
                        ;
                    } else {
                        // payment processing is complete, or there is no payment processor transaction
                        result = PostStatus.postingAllowed();
                    }
                }
            } else {
                result = PostStatus.postingAllowed();
            }
        }
        return result;
    }

    /**
     * Determines if a payment or refund has an outstanding EFT transaction.
     * <p/>
     * This is true for any:
     * <ul>
     *     <li><em>act.customerAccountPaymentEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSPayment</em></li>
     *     <li><em>act.customerAccountRefundEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSRefund</em></li>
     * </ul>
     * NOTE that this doesn't apply to payments and refunds where there is no integrated EFTPOS support; at least
     * one <em>act.EFTPOSPayment</em> or <em>act.EFTPOSRefund</em> must be present.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is DECLINED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding EFT transaction.
     */
    public boolean hasOutstandingEFTPOSTransaction(FinancialAct act, boolean ignoreUnsuccessful) {
        return getEFTPOSTransactionStatus(act, ignoreUnsuccessful).getStatus() == ItemStatus.Status.INCOMPLETE;
    }

    /**
     * Posts a customer account act.
     * <p/>
     * This sets its status to {@link FinancialActStatus#POSTED}, and sets the
     * {@link FinancialAct#setActivityStartTime(Date) start time} to now. The latter is to ensure that transactions
     * done prior to an em>act.customerAccountClosingBalance</em> and subsequently POSTED don't are counted in the
     * current balance.
     *
     * @param act the act to post
     * @throws CustomerAccountRuleException if the act cannot be posted
     */
    public void post(FinancialAct act) throws CustomerAccountRuleException {
        if (!act.isA(DEBITS_CREDITS)) {
            throw new CustomerAccountRuleException(CannotPostAct, DescriptorHelper.getDisplayName(act, service));
        } else if (POSTED.equals(act.getStatus())) {
            throw new CustomerAccountRuleException(AlreadyPosted, DescriptorHelper.getDisplayName(act, service));
        }
        TransactionTemplate template = new TransactionTemplate(transactionManager);
        template.execute(new TransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) {
                if (act.isA(PAYMENT, REFUND)) {
                    if (hasOutstandingEFTPOSTransaction(act, false)) {
                        throw new CustomerAccountRuleException(CannotPostWithOutstandingEFT,
                                                               DescriptorHelper.getDisplayName(act, service));
                    } else if (hasOutstandingPaymentProcessorItem(act, false)) {
                        String itemDisplayName = act.isA(PAYMENT)
                                                 ? DescriptorHelper.getDisplayName(PAYMENT_PP, service)
                                                 : DescriptorHelper.getDisplayName(REFUND_PP, service);
                        throw new CustomerAccountRuleException(CannotPostWithOutstandingPaymentProcessorItem,
                                                               DescriptorHelper.getDisplayName(act, service),
                                                               itemDisplayName);
                    }
                }
                act.setStatus(POSTED);
                act.setActivityStartTime(new Date()); // see OVPMS-734
                ruleService.save(act);
            }
        });
    }

    /**
     * Determines if a payment or refund has an EFTPOS transaction with APPROVED status.
     *
     * @param act the payment/refund
     * @return {@code true} if at least one transaction is APPROVED, otherwise {@code false}
     */
    public boolean hasApprovedEFTPOSTransaction(FinancialAct act) {
        boolean result = false;
        List<FinancialAct> items = getEFTItems(act);
        for (FinancialAct item : items) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest != null && EFTPOSTransactionStatus.APPROVED.equals(latest.getStatus())) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Determines if a payment or refund has an outstanding payment processor item.
     * <p/>
     * This is any payment processor item without a COMPLETED transaction.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is CANCELLED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding payment processor transaction,
     * otherwise {@code false}
     */
    public boolean hasOutstandingPaymentProcessorItem(FinancialAct act, boolean ignoreUnsuccessful) {
        ItemStatus status = getPaymentProcessorTransactionStatus(act, ignoreUnsuccessful);
        return status.isUnsubmitted() || status.isIncomplete();
    }

    /**
     * Determines if a payment or refund has a Payment Processor transaction with COMPLETED status.
     *
     * @param act the payment/refund
     * @return {@code true} if at least one transaction is COMPLETED, otherwise {@code false}
     */
    public boolean hasCompletedPaymentProcessorItem(FinancialAct act) {
        boolean result = false;
        for (FinancialAct item : getPaymentProcessorItems(act)) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest != null && PaymentProcessorTransactionStatus.COMPLETED.equals(latest.getStatus())) {
                result = true;
                break;
            }
        }
        return result;
    }

    /**
     * Returns the latest payment processor transaction from a payment or refund.
     * <p/>
     * Payments/refunds may only have a single payment processor item.
     *
     * @param act the payment/refund
     * @return the most recent payment processor transaction, or {@code null} if there is none
     */
    public FinancialAct getPaymentProcessorTransaction(FinancialAct act) {
        FinancialAct result = null;
        IMObjectBean bean = service.getBean(act);
        Reference targetRef = bean.getTargetRef("items",
                                                Policies.match(true, Predicates.targetIsA(PAYMENT_PP, REFUND_PP)));
        if (targetRef != null) {
            FinancialAct item = bean.getObject(targetRef, FinancialAct.class);
            if (item != null) {
                result = getMostRecentTransaction(item);
            }
        }
        return result;
    }

    private static class ItemStatus {

        public enum Status {
            NO_TRANSACTION, // No transaction present
            UNSUBMITTED,    // Transaction needs to be submitted
            INCOMPLETE,     // Transaction submitted, but not completed
            COMPLETE,       // Transaction is complete
        }

        private final Status status;

        private final FinancialAct item;

        private final FinancialAct transaction;

        public ItemStatus(Status status) {
            this(status, null, null);
        }

        public ItemStatus(Status status, FinancialAct item, FinancialAct transaction) {
            this.status = status;
            this.item = item;
            this.transaction = transaction;
        }

        public boolean isUnsubmitted() {
            return status == Status.UNSUBMITTED;
        }

        public boolean isIncomplete() {
            return status == Status.INCOMPLETE;
        }

        public boolean isComplete() {
            return status == Status.COMPLETE;
        }

        public Status getStatus() {
            return status;
        }

        public FinancialAct getItem() {
            return item;
        }

        public FinancialAct getTransaction() {
            return transaction;
        }
    }

    /**
     * Determines if a payment or refund has an outstanding EFT transaction.
     * <p/>
     * This is true for any:
     * <ul>
     *     <li><em>act.customerAccountPaymentEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSPayment</em></li>
     *     <li><em>act.customerAccountRefundEFT</em> that doesn't have an APPROVED or NO_TERMINAL
     *     <em>act.EFTPOSRefund</em></li>
     * </ul>
     * NOTE that this doesn't apply to payments and refunds where there is no integrated EFTPOS support; at least
     * one <em>act.EFTPOSPayment</em> or <em>act.EFTPOSRefund</em> must be present.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is DECLINED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding EFT transaction.
     */
    private ItemStatus getEFTPOSTransactionStatus(FinancialAct act, boolean ignoreUnsuccessful) {
        ItemStatus result = new ItemStatus(ItemStatus.Status.NO_TRANSACTION);
        for (FinancialAct item : getEFTItems(act)) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest == null) {
                // offline EFT items don't have a transaction
                break;
            } else {
                String status = latest.getStatus();
                if (EFTPOSTransactionStatus.PENDING.equals(status)
                    || EFTPOSTransactionStatus.IN_PROGRESS.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                } else if (EFTPOSTransactionStatus.APPROVED.equals(status)
                           || EFTPOSTransactionStatus.NO_TERMINAL.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.COMPLETE, item, latest);
                } else if (!ignoreUnsuccessful) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                }
            }
        }
        return result;
    }

    /**
     * Determines if a payment or refund has an outstanding payment processor item.
     * <p/>
     * This is any payment processor item without a COMPLETED transaction.
     *
     * @param act                the payment or refund
     * @param ignoreUnsuccessful if {@code true}, ignore any transaction that is CANCELLED, or ERROR
     * @return {@code true} if a payment or refund has an outstanding payment processor transaction,
     * otherwise {@code false}
     */
    private ItemStatus getPaymentProcessorTransactionStatus(FinancialAct act, boolean ignoreUnsuccessful) {
        ItemStatus result = new ItemStatus(ItemStatus.Status.NO_TRANSACTION);
        for (FinancialAct item : getPaymentProcessorItems(act)) {
            FinancialAct latest = getMostRecentTransaction(item);
            if (latest == null) {
                result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, null);
                break;
            } else {
                String status = latest.getStatus();
                if (PaymentProcessorTransactionStatus.PENDING.equals(status)
                    || PaymentProcessorTransactionStatus.IN_PROGRESS.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, latest);
                    break;
                } else if (PaymentProcessorTransactionStatus.SUBMITTED.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.INCOMPLETE, item, latest);
                    break;
                } else if (PaymentProcessorTransactionStatus.COMPLETED.equals(status)) {
                    result = new ItemStatus(ItemStatus.Status.COMPLETE);
                } else if (!ignoreUnsuccessful) {
                    result = new ItemStatus(ItemStatus.Status.UNSUBMITTED, item, latest);
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Returns the {@link CustomerAccountArchetypes#PAYMENT_EFT} or {@link CustomerAccountArchetypes#REFUND_EFT}
     * items associated with a payment or refund.
     *
     * @param act the payment/refund
     * @return the corresponding EFT items
     */
    private List<FinancialAct> getEFTItems(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.getTargets(ITEMS, FinancialAct.class,
                               Policies.all(Predicates.targetIsA(PAYMENT_EFT, REFUND_EFT)));
    }

    /**
     * Returns the most recent transaction from an item (i.e. an EFT or payment processor payment/refund).
     *
     * @param item the item
     * @return the most recent transaction, or {@code null} if there are no transactions
     */
    private FinancialAct getMostRecentTransaction(FinancialAct item) {
        List<FinancialAct> transactions = service.getBean(item)
                .getTargets("transactions", FinancialAct.class, Policies.orderBySequence(false));
        return !transactions.isEmpty() ? transactions.get(0) : null;
    }

    /**
     * Returns the {@link CustomerAccountArchetypes#PAYMENT_PP} or {@link CustomerAccountArchetypes#REFUND_PP}
     * items associated with a payment or refund.
     *
     * @param act the payment/refund
     * @return the corresponding payment processor items
     */
    private List<FinancialAct> getPaymentProcessorItems(FinancialAct act) {
        IMObjectBean bean = service.getBean(act);
        return bean.getTargets(ITEMS, FinancialAct.class, Policies.all(Predicates.targetIsA(PAYMENT_PP, REFUND_PP)));
    }

    /**
     * Creates an opening/closing balance act.
     *
     * @param archetype the act archetype
     * @param customer  the customer
     * @param date      the date
     * @param amount    the total amount. Nay be negative
     * @return a new act
     */
    private FinancialAct createBalance(String archetype, Party customer, Date date, BigDecimal amount) {
        FinancialAct act = service.create(archetype, FinancialAct.class);
        act.setActivityStartTime(date);
        if (amount.signum() == -1) {
            amount = amount.negate();
            act.setCredit(!act.isCredit());
        }
        act.setTotal(amount);
        IMObjectBean bean = service.getBean(act);
        bean.setTarget(CUSTOMER, customer);
        return act;
    }

    /**
     * Removes charge items and medications acts linked to an invoice from the patient history.
     *
     * @param invoice the invoice
     * @param toSave  a list of objects to save
     */
    private void removeInvoiceFromPatientHistory(FinancialAct invoice, List<IMObject> toSave) {
        IMObjectBean bean = service.getBean(invoice);
        Map<Reference, Act> events = new HashMap<>();
        for (Act item : bean.getTargets(ITEMS, Act.class)) {
            IMObjectBean itemBean = service.getBean(item);
            ActRelationship relationship = itemBean.getObject("event", ActRelationship.class);
            if (relationship != null) {
                toSave.add(item);
                removeEventRelationship(events, item, relationship);
            }
            for (Act medication : itemBean.getTargets("dispensing", Act.class)) {
                if (removeEventRelationship(events, medication)) {
                    toSave.add(medication);
                }
            }
            for (Act investigation : itemBean.getTargets("investigations", Act.class)) {
                if (removeEventRelationship(events, investigation)) {
                    toSave.add(investigation);
                }
            }
            for (Act document : itemBean.getTargets("documents", Act.class)) {
                if (removeEventRelationship(events, document)) {
                    toSave.add(document);
                }
            }
        }
        toSave.addAll(events.values());
    }

    /**
     * Removes a relationship between an act and <em>act.patientClinicalEvent</em>.
     *
     * @param events the cache of events
     * @param act    the act to remove the relationship from
     */
    private boolean removeEventRelationship(Map<Reference, Act> events, Act act) {
        boolean changed = false;
        IMObjectBean bean = service.getBean(act);
        for (ActRelationship eventRelationship : bean.getValues("event", ActRelationship.class)) {
            changed = true;
            removeEventRelationship(events, act, eventRelationship);
        }
        return changed;
    }

    /**
     * Removes a relationship between an act and <em>act.patientClinicalEvent</em>.
     *
     * @param events       the cache of events
     * @param act          the act to remove the relationship from. It must be the target of the relationship
     * @param relationship the relationship to remove
     */
    private void removeEventRelationship(Map<Reference, Act> events, Act act, ActRelationship relationship) {
        act.removeActRelationship(relationship);
        Reference ref = relationship.getSource();
        Act event = events.computeIfAbsent(ref, reference -> service.get(reference, Act.class));
        if (event != null) {
            event.removeActRelationship(relationship);
        }
    }

    /**
     * Return the latest charge for a customer.
     *
     * @param shortName the charge archetype short name
     * @param customer  the customer
     * @return the invoice, or {@code null} if none can be found
     */
    private FinancialAct getCharge(String shortName, Reference customer) {
        FinancialAct result = getCharge(shortName, customer, IN_PROGRESS);
        if (result == null) {
            result = getCharge(shortName, customer, ActStatus.COMPLETED);
        }
        return result;
    }

    /**
     * Return the latest charge for a customer with the given status.
     *
     * @param shortName the charge archetype short name
     * @param customer  the customer
     * @param status    the act status
     * @return the invoice, or {@code null} if none can be found
     */
    private FinancialAct getCharge(String shortName, Reference customer, String status) {
        ArchetypeQuery query = new ArchetypeQuery(shortName, false, true);
        query.setMaxResults(1);

        query.add(Constraints.join(CUSTOMER).add(Constraints.eq("entity", customer)));
        query.add(Constraints.eq(STATUS, status));
        query.add(Constraints.sort(START_TIME, false));
        IMObjectQueryIterator<FinancialAct> iterator = new IMObjectQueryIterator<>(service, query);
        return iterator.hasNext() ? iterator.next() : null;
    }

}
